#ifndef _TASK_H_
#define _TASK_H_

#include "mbed.h"

/**
 * @author Christian Burri
 *
 * @copyright Copyright (c) 2013 HSLU Pren Team #1 Cruising Crêpe
 * All rights reserved.
 *
 * @brief
 * The <code>Task</code> class allows to install periodic, time-triggered
 * tasks. An example of a simple user-defined task is given below:
 * <pre><code>
 * class MyTask : public Task {
 *   public:
 *     void run();
 * };
 *
 * void MyTask::run() {
 *   <span style="color:#FF0000">// code to be executed periodically</span>
 * }
 * </code></pre>
 * This task can then be created and started as follows:
 * <pre><code>
 * MyTask myTask(0.1);    <span style="color:#FF0000">// period in seconds</span>
 * myTask.start();
 * ...
 *
 * myTask.stop();
 * </code></pre>
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
class Task
{

private:

    /** specifiying the interval in seconds */
    float       period;
    /** The Ticker interface is used to setup a recurring interrupt to
     *  repeatedly call a function at a specified rate.
     */
    Ticker      ticker;

public:

    /**
     * Creates a task object with a given period.
     * @param period the period of this task in seconds.
     */
    Task(float period);

    virtual ~Task();

    /**
     * Gets the period of this task.
     * @return the period in seconds.
     */
    float           getPeriod();

    /**
     * Starts this task.
     */
    void            start();

    /**
     * Stops this task.
     */
    void            stop();

    /**
     * This method needs to be implemented by a user task.
     */
    virtual void    run();
};

#endif
