#ifndef _STATE_H_
#define _STATE_H_

#include "MaxonESCON.h"
#include "RobotControl.h"
#include "Task.h"
#include "defines.h"

/**
 * @author Christian Burri
 *
 * @copyright Copyright (c) 2013 HSLU Pren Team #1 Cruising Crêpe
 * All rights reserved.
 *
 * @brief
 *
 * State is the main mechanism for communicating current realtime system state to
 * the rest of the system for logging etc.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

class State : public Task
{

private:

    state_t*            s;
    RobotControl*       robotControl;
    MaxonESCON*         motorControllerLeft;
    MaxonESCON*         motorControllerRight;
    AnalogIn            battery;
    Timer               timer;
    float               period;

public:

    /**
     * @brief Creates a robot control object and initializes all private state variables.
     * @param s struct to read and write the state
     * @param robotControl Object to read the state
     * @param motorControllerLeft a reference to the servo drive for the left motor
     * @param motorControllerRight a reference to the servo drive for the right motor
     * @param batteryPin mbed pin for analog Input Battery Voltage
     * @param period the sampling period of the run loop of this controller, given in [s]
     */
    State(state_t* s,
          RobotControl* robotControl,
          MaxonESCON* motorControllerLeft,
          MaxonESCON* motorControllerRight,
          PinName batteryPin,
          float period);

    /**
     * @brief Destructor of the Object to destroy the Object.
     */
    virtual     ~State();

    /**
     * @brief Initzialize the File. Open the File plots.txt and set the title at first line.
     */
    void initPlotFile(void);

    /**
     * @brief Close the File.
     */
    void closePlotFile(void);

    /**
     * @brief Return the Battery voltage
     * state variables.
     * @return Batterie Voltage [V]
     */
    float readBattery();

    /**
     * @brief Starts the timer from zero.
     * The timer is for the logging Time.
     */
    void startTimerFromZero();

    /**
     * @brief Save the new state to a new line.
     */
    void savePlotFile(state_t s);
    
    /**
    * Returns the transform a  sinusvalue from -1.0 .. 1.0 to 0.0 .. 1.0
    * @param idx number of LEDS
    * @param f value till 2&pi;
    * @return sinus value from 0.0f 1.0f
    */
    float dim( int idx, float f );

    /**
     * @brief Run method actualize every period.
     */
    void        run();

private:

    void setBatteryBit();

    void setEnableLeftBit();

    void setEnableRightBit();
};

#endif
