/*
Developer: Christian Andresen

Project name: Project Greenhouse

Brief description:
    Microcontroller observing the climate within a greenhouse, and displaying warnings/information as needed.
    Sensors monitor temperature and humidity.
    To simulate an actual greenhouse, LEDs will be used in place of actual enviromental influences.
    Simulator mode has been added for debugging and presentation purposes.
*/

#include "mbed.h" // Mbed header from mbed-os version 5
#include "stm32746g_discovery_lcd.h" // Last updated 27/11/2019
#include "DHT.h" // Author: Wim De Roeve
#include <iostream>
#include "bootup.h" // Author: Christian Andresen

// Call Bootup class from bootup.h as LCD_Start
bootup LCD_Start;

/* Inputs */
AnalogIn Temp_Sensor(A0);
DHT Humid_Sensor(A1, DHT22);
InterruptIn simbutton(D2);

/* Outputs */
DigitalOut Temp_LED(D3); // Red LED
DigitalOut Humid_LED(D4); // Blue LED
DigitalOut Heat_LED(D5); // Red LED
DigitalOut Water_LED(D6); // Blue LED

/* Multithreads */
Thread tTimeCounter;
Thread tLocation_Select;
Thread tTemp;
Thread tHumid;
Thread tTemp_Check;
Thread tHumid_Check;

/* SIMULATION MODE */
bool simmode = 1;
void Sim_Toggle();

/* Device location */
void Location_Select();
int location_id;

/* Day & Night */
void TimeCounter();
void DayMode();
void NightMode();
int timecount = 0;
bool timeofday = 1;

/* Temp and Humidity */
void Temp_Display(); // Displays temperatures in Celsius on screen
void Humid_Display(); // Displays humidity in % on screen
void Temp_Check();
void Humid_Check();
float temp;
float humid;

/* Greenhouse mechanisms */
void Vent_Air();
void Heat_Lamp();
void Dehumidifier();
void Spray_Water();

uint8_t text[30]; // Used to display values with added text.
uint8_t text2[30]; // Used to display values with added text.

void startup()
{
    cout << "Enter Greenhouse ID: \n \r";
    cin >> location_id;
}

int main()
{
    startup();
    LCD_Start.LCD_Bootup();
    tTimeCounter.start(&TimeCounter);
    tLocation_Select.start(&Location_Select);
    simbutton.rise(&Sim_Toggle);
    tTemp.start(&Temp_Display);
    tHumid.start(&Humid_Display);
    tTemp_Check.start(&Temp_Check);
    tHumid_Check.start(&Humid_Check);
    while(1) {}
}

void Sim_Toggle()
{
    simmode = !simmode;
}

void Location_Select()
{
    while(1) {
        sprintf((char*)text2, "Greenhouse %d", location_id);
        BSP_LCD_DisplayStringAt(0, 0, (uint8_t *)&text2, LEFT_MODE);
        if(simmode) {
            BSP_LCD_DisplayStringAt(0, 25, (uint8_t *) "SIM MODE", LEFT_MODE);
        } else if(simmode == 0) {
            BSP_LCD_DisplayStringAt(0, 25, (uint8_t *) "        ", LEFT_MODE);
        }
    }
}

void TimeCounter()
{
    // Timecounter counts up to one cycle of 
    // 720 minutes if not in Simulation mode
    while(1) {
        if(simmode) {
            timecount++;
            wait(0.01); // One tenth of a second per tic
            if (timecount == 721) {
                timeofday = !timeofday;
                printf("Time of day is %d \n \r", timeofday);
                timecount = 0;
                if (timeofday) {
                    DayMode();
                } else {
                    NightMode();
                }
            }
        } else if(simmode == 0) {
            timecount++;
            wait(60); // 1 minute per tic
            if (timecount == 721) {
                timeofday = !timeofday;
                timecount = 0;
                if (timeofday) {
                    DayMode();
                } else {
                    NightMode();
                }
            }
        }
    }
}

void DayMode()
{
    BSP_LCD_Clear(LCD_COLOR_WHITE);
    BSP_LCD_SetBackColor(LCD_COLOR_WHITE);
    BSP_LCD_SetTextColor(LCD_COLOR_BLACK);
    BSP_LCD_DisplayStringAt(0, 0, (uint8_t *) "  Time: Day  ", RIGHT_MODE);
}

void NightMode()
{
    BSP_LCD_Clear(LCD_COLOR_BLACK);
    BSP_LCD_SetBackColor(LCD_COLOR_BLACK);
    BSP_LCD_SetTextColor(LCD_COLOR_WHITE);
    BSP_LCD_DisplayStringAt(0, 0, (uint8_t *) " Time: Night ", RIGHT_MODE);
}

void Temp_Display()
{
    while(1) {
        // Monitor temperatures and display results on LCD screen
        sprintf((char*)text, "Temperature: %.1f C   ", temp);
        BSP_LCD_DisplayStringAt(0, 150, (uint8_t *)&text, CENTER_MODE);
        wait(1);
    }
}

void Humid_Display()
{
    while(1) {
        // Monitor humidity and display results on LCD screen
        sprintf((char*)text, "Humidity: %.1f   ", humid);
        BSP_LCD_DisplayStringAt(0, 200, (uint8_t *)&text, CENTER_MODE);
        wait(1);
    }
}

void Temp_Check()
{
    while(1) {
        temp = Temp_Sensor.read() * 100 / 2; 
        // Read temperature sensor input. Since this is flawed, correct to adjust.
        if(timeofday) { 
        // If it's daytime, keep temps between 25 and 30 celsius
            if (temp >= 25 && temp <= 30) {
                Temp_LED = 1;
            }  else if (temp < 25) {
                Temp_LED = 0;
                Heat_Lamp(); 
                // Raise temperature by turning on the heat lamps in the greenhouse
            } else if (temp > 30) {

                Temp_LED = 0;
                Vent_Air(); 
                // Reduce temperature by venting the hot air out of the greenhouse
            }
        } else if(timeofday == false) { 
        // If it's nighttime, keep temps between 15 and 20
            if (temp >= 15 && temp <= 20) {
                Temp_LED = 1;
            } else if (temp < 15) {
                Temp_LED = 0;
                Heat_Lamp();

            } else if (temp > 20) {
                Temp_LED = 0;
                Vent_Air();
            }
        }
    }
}

void Humid_Check()
{
    while(1) {
        humid = Humid_Sensor.ReadHumidity();
        int dataread = Humid_Sensor.readData(); // No idea why, but the humidity won't display on screen without it.
        if (humid >= 55 && humid <= 65) { // Check if humidity is within acceptable parameters (55% - 65%)
            Humid_LED = 1;
        } else if (humid < 55) {
            Humid_LED = 0;
            Spray_Water(); // Raise humidity by spraying water into the plants and greenhouse.
        } else if ( humid > 65) {
            Humid_LED = 0;
            Dehumidifier(); // Lower humidity by dehumidifying the greenhouse.
        }
    }
}

void Vent_Air()
{
    Heat_LED = 1;
    wait(0.5);
    Heat_LED = 0;
    if(simmode) {
        printf("Greenhouse temperature above sustainable levels. Venting air from Greenhouse to rectify. \n \r");
    }
}

void Heat_Lamp()
{
    Heat_LED = 1;
    wait(1);
    Heat_LED = 0;
    if(simmode) {
        printf("Greenhouse temperature below acceptable levels. Turning on heat lamps to rectify. \n \r");
    }
}

void Dehumidifier()
{
    Water_LED = 1;
    wait(0.5);
    Water_LED = 0;
    if(simmode) {
        printf("Greenhouse humidity above acceptable levels. Dehumidifying to rectify. \n \r");
    }
}

void Spray_Water()
{
    Water_LED = 1;
    wait(1);
    Water_LED = 0;
    if(simmode) {
        printf("Greenhouse humidity below acceptable levels. Spraying water to rectify. \n \r");
    }
}