/*
    Author: Christian Andresen
    Website: xcha11.skp-dp.sde.dk
    Date: 10-01-2020
    Brief: STM32-F746NG-DISCO Project
    Project includes:
        Button/Touch sensor input, connected to LED lights and grove buzzer.
        Infinite counter of 0-9999. Resets automatically.
        Blue screen of death
        Multithreading
        Debug statements located within vital processes, for easy checking over console.

*/
#include "mbed.h" // mbed-os 5
#include "stm32746g_discovery_lcd.h" // Last updated 27/11/2019

// D2-D4: Input, D5-D8: Output.
InterruptIn button(D2);
InterruptIn touch(D3);
DigitalOut led(D5);
DigitalOut buzzer(D6);

// Multithreading
Thread CounterThread;
Thread InputThread;

// Prototypes
void bluescreen();
void input();
void buttonpress();
void touchpress();

// Button click counter, button-press bool, and determinator.
int button_count;
bool buttonclick = false;

// Touch count, touch-sensor bool, and determinator
int touch_count;
bool touchclick = false;

// Integers used for the 9999 counter.
int countup;
int restarts;

// 9999 counter program. Added above main for multithread ability.
void ninecounter()
{
    HAL_Delay(10000);
    while(1) {
        if (countup != 9999) {
            wait(0.001);
            countup++;

        } else if (restarts == 10) {
            break; // Once BSOD hits, the break statement will end the 9999 counter to simulate proper system shutdown.
        } else {
            countup = 0; // Reset 9999 counter
            restarts++; // Add to restart
            printf("Counter has finished counting. Reset nr. %d. \r\n", restarts);
        }
    }
}

// Main part of the program
int main()
{
    // Start multithread ninecounter & inputcounter
    CounterThread.start(&ninecounter);
    InputThread.start(&input);

    // Checks when the button is pushed down and when the touch sensor is activated
    button.rise(&buttonpress);
    touch.rise(&touchpress);

    uint8_t text[30]; // Prepare text element for later print to LCD
    button_count = 0; // Amount of times the button has been pressed
    touch_count = 0; // Amount of times the touch sensor has been activated
    countup = 0; // Default value for the counter
    restarts = 0; // Amount of times 9999 counter has been reset

    // Boot up LCD screen
    BSP_LCD_Init(); // Starts up the LCD
    BSP_LCD_LayerDefaultInit(LTDC_ACTIVE_LAYER, LCD_FB_START_ADDRESS);
    BSP_LCD_SelectLayer(LTDC_ACTIVE_LAYER);

    // Clear the screen, add black background and red text
    BSP_LCD_Clear(LCD_COLOR_BLACK);
    BSP_LCD_SetBackColor(LCD_COLOR_BLACK);
    BSP_LCD_SetTextColor(LCD_COLOR_RED);

    // Bootup Message
    printf("Device is booting up \r\n"); // Debug statement that lets you know that the device is booting.
    BSP_LCD_DisplayStringAt(0, 100, (uint8_t *) "HELLDROP STUDIO", CENTER_MODE); // Company
    BSP_LCD_DisplayStringAt(0, 150, (uint8_t *) "A big hard business", CENTER_MODE); // Tagline 1
    BSP_LCD_DisplayStringAt(0, 175, (uint8_t *) "in a big hard building", CENTER_MODE); // Tagline 2
    HAL_Delay(10000); // Delay 10 seconds to simulate actual bootup

    // Change LCD Screen to system
    BSP_LCD_Clear(LCD_COLOR_BLACK);
    printf("Device has started functioning \r\n");

    BSP_LCD_DisplayStringAt(0, 25, (uint8_t *) "HELLDROP STUDIO SOFTWARE", CENTER_MODE); // Studio name & department
    BSP_LCD_DisplayStringAt(0, 250, (uint8_t *) "Code by Christian Andresen", CENTER_MODE); // Developer credit

    // While there are less than 10 restarts of the counter..
    while(restarts < 10) {
        // Combine string and integer value.
        sprintf((char*)text, "Button count: %d", button_count);
        // Print value to LCD screen
        BSP_LCD_DisplayStringAt(0, 100, (uint8_t *)&text, CENTER_MODE);
        sprintf((char*)text, "Touch count: %d", touch_count);
        BSP_LCD_DisplayStringAt(0, 150, (uint8_t *)&text, CENTER_MODE);
        sprintf((char*)text, "   %d  (%d) ", countup, restarts);
        BSP_LCD_DisplayStringAt(0, 200, (uint8_t *)&text, CENTER_MODE);
    }
    if (restarts == 10) { // If there are 10 restarts or more, trigger BSOD
        bluescreen();
    }
}

void bluescreen()
{
    //BLUE SCREEN OF DEATH
    printf("FATAL ERROR. SYSTEM REQUIRES MANUAL REBOOT \r\n");
    BSP_LCD_Clear(LCD_COLOR_BLUE);
    BSP_LCD_SetBackColor(LCD_COLOR_BLUE);
    BSP_LCD_SetTextColor(LCD_COLOR_WHITE);
    BSP_LCD_DisplayStringAt(0, 75, (uint8_t *) "This device has", CENTER_MODE);
    BSP_LCD_DisplayStringAt(0, 100, (uint8_t *) "encountered", CENTER_MODE);
    BSP_LCD_DisplayStringAt(0, 125, (uint8_t *) "a fatal error", CENTER_MODE);
    BSP_LCD_DisplayStringAt(0, 150, (uint8_t *) "and needs to reboot.", CENTER_MODE);
    BSP_LCD_DisplayStringAt(0, 250, (uint8_t *) "HAL Error", LEFT_MODE);
    BSP_LCD_DisplayStringAt(0, 250, (uint8_t *) "ERROR CODE: 40", RIGHT_MODE);
}

void input()
{
    HAL_Delay(10000);
    while(1) {
        if(buttonclick == true) {
            printf("A button has been pressed. \r\n");
            button_count++; // Add to button counter
            led = 1; // Turn on LED
            buzzer = 1; // Turn on buzzer
            wait(0.1); // Prevent immediate shutdown
            // Turn both off
            led = 0;
            buzzer = 0;
            buttonclick = false; // End button click
        } else if(touchclick == true) {
            printf("Touch sensor has been activated \r\n");
            touch_count++; // Add to touch counter on the LCD
            led = 1;
            wait(0.1);
            led = 0;
            touchclick = false; // End touch click
        } else if (restarts == 10) {
            break; // Once BSOD hits, the break statement will end the input counter to simulate proper system shutdown.
        } else { // Failsafe if neither are turned off during click.
            led = 0;
            buzzer = 0;
        }
    }
}

void buttonpress()
{
    buttonclick = true; // Bool that activates the input() once it registers a button press
}

void touchpress()
{
    touchclick = true; // See Buttonpress() comments above
}