#include "cdms_sd.h"

SPI spi_sd(PTE1, PTE3, PTE2);      // MOSI,MISO, CLOCK microcontroller(in order)     
DigitalOut cs_sd(PTE22);

Serial sd1(USBTX,USBRX);

int cdv;
uint64_t sd_sectors();
uint64_t sectors;

void FCTN_SD_MNGR()
{
    /*Size of block of SD card for 2GB = 512B, 4 , 8 GB SD card. We will prefer 8 GB.
    SD allocation. Assuming 8GB
    SCP: 600 MB -122880
    SFF-AT: 2 GB -4194304
    SFF-BT: 5 GB -10485760
    HK-ARCH:100 MB -204800
    LOG: 50MB -102400
    SD card management: 50MB - 102400*/
       
}
int initialise_card()
{
    // Set to 100kHz for initialisation, and clock card with cs_sd = 1
    spi_sd.frequency(100000);
    cs_sd = 1;
    for (int i = 0; i < 16; i++) {
        spi_sd.write(0xFF);
    }

    // send CMD0, should return with all zeros except IDLE STATE set (bit 0)
    if (cmd(0, 0) != R1_IDLE_STATE) {
        debug("No disk, or could not put SD card in to spi_sd idle state\r\n");
        return SDCARD_FAIL;
    }

// send CMD8 to determine whther it is ver 2.x
    int r = cmd8();
    if (r == R1_IDLE_STATE) {
        printf("\rEntering v2\r\n");
        return initialise_card_v2();

    } else if (r == (R1_IDLE_STATE | R1_ILLEGAL_COMMAND)) {
        printf("\rEntering v1\r\n");
        return initialise_card_v1();

    } else {
        debug("\rNot in idle state after sending CMD8 (not an SD card?)\r\n");
        return SDCARD_FAIL;
    }
}

int initialise_card_v1()
{
    for (int i = 0; i < SD_COMMAND_TIMEOUT; i++) {
        cmd(55, 0);
        if (cmd(41, 0) == 0) {
            printf("\rv1 initialization successfull\r\n");
            cdv = 512;
            debug_if(SD_DBG, "\n\rInit: SEDCARD_V1\n\r");

            return SDCARD_V1;
        }
    }

    debug("\rTimeout waiting for v1.x card\r\n");
    return SDCARD_FAIL;
}


int initialise_card_v2()
{
    for (int i = 0; i < SD_COMMAND_TIMEOUT; i++) {
        wait_ms(50);
        cmd58();
        cmd(55, 0);
        if (cmd(41, 0x40000000) == 0) {
            printf("\rv2 initialization successfull\r\n");
            cmd58();
            debug_if(SD_DBG, "\n\rInit: SDCARD_V2\n\r");
            cdv = 1;

            return SDCARD_V2;
        }
    }

    debug("\rTimeout waiting for v2.x card\r\n");
    return SDCARD_FAIL;
}

int cmd(int cmd, int arg)
{
    cs_sd = 0;

    // send a command
    spi_sd.write(0x40 | cmd);
    spi_sd.write(arg >> 24);
    spi_sd.write(arg >> 16);
    spi_sd.write(arg >> 8);
    spi_sd.write(arg >> 0);
    spi_sd.write(0x95);

    // wait for the repsonse (response[7] == 0)
    for (int i = 0; i < SD_COMMAND_TIMEOUT; i++) {
        int response = spi_sd.write(0xFF);
        if (!(response & 0x80)) {
            cs_sd = 1;
            spi_sd.write(0xFF);
            return response;
        }
    }
    cs_sd = 1;
    spi_sd.write(0xFF);
    return -1; // timeout
}


int cmd58()
{
    cs_sd = 0;
    int arg = 0;

    // send a command
    spi_sd.write(0x40 | 58);
    spi_sd.write(arg >> 24);
    spi_sd.write(arg >> 16);
    spi_sd.write(arg >> 8);
    spi_sd.write(arg >> 0);
    spi_sd.write(0x95);

    // wait for the repsonse (response[7] == 0)
    for (int i = 0; i < SD_COMMAND_TIMEOUT; i++) {
        int response = spi_sd.write(0xFF);
        if (!(response & 0x80)) {
            int ocr = spi_sd.write(0xFF) << 24;
            ocr |= spi_sd.write(0xFF) << 16;
            ocr |= spi_sd.write(0xFF) << 8;
            ocr |= spi_sd.write(0xFF) << 0;
            cs_sd = 1;
            spi_sd.write(0xFF);
            return response;
        }
    }
    cs_sd = 1;
    spi_sd.write(0xFF);
    return -1; // timeout
}


int cmd8()
{
    cs_sd = 0;

    // send a command
    spi_sd.write(0x40 | 8); // CMD8
    spi_sd.write(0x00);     // reserved
    spi_sd.write(0x00);     // reserved
    spi_sd.write(0x01);     // 3.3v
    spi_sd.write(0xAA);     // check pattern
    spi_sd.write(0x87);     // crc

    // wait for the repsonse (response[7] == 0)
    for (int i = 0; i < SD_COMMAND_TIMEOUT * 1000; i++) {
        char response[5];
        response[0] = spi_sd.write(0xFF);
        if (!(response[0] & 0x80)) {
            for (int j = 1; j < 5; j++) {
                response[i] = spi_sd.write(0xFF);
            }
            cs_sd = 1;
            spi_sd.write(0xFF);
            return response[0];
        }
    }
    cs_sd = 1;
    spi_sd.write(0xFF);
    return -1; // timeout
}

uint64_t sd_sectors()
{
    uint32_t c_size, c_size_mult, read_bl_len;
    uint32_t block_len, mult, blocknr, capacity;
    uint32_t hc_c_size;
    uint64_t blocks;

    // CMD9, Response R2 (R1 byte + 16-byte block read)
    if (cmdx(9, 0) != 0) {
        debug("\rDidn't get a response from the disk\n");
        return 0;
    }

    uint8_t cs_sdd[16];
    if (read(cs_sdd, 16) != 0) {
        debug("\rCouldn't read cs_sdd response from disk\n");
        return 0;
    }

    // cs_sdd_structure : cs_sdd[127:126]
    // c_size        : cs_sdd[73:62]
    // c_size_mult   : cs_sdd[49:47]
    // read_bl_len   : cs_sdd[83:80] - the *maximum* read block length

    int cs_sdd_structure = ext_bits(cs_sdd, 127, 126);

    switch (cs_sdd_structure) {
        case 0:
            cdv = 512;
            c_size = ext_bits(cs_sdd, 73, 62);
            c_size_mult = ext_bits(cs_sdd, 49, 47);
            read_bl_len = ext_bits(cs_sdd, 83, 80);

            block_len = 1 << read_bl_len;
            mult = 1 << (c_size_mult + 2);
            blocknr = (c_size + 1) * mult;
            capacity = blocknr * block_len;
            blocks = capacity / 512;
            debug_if(SD_DBG, "\n\rSDCard\n\rc_size: %d \n\rcapacity: %ld \n\rsectors: %lld\n\r", c_size, capacity, blocks);
            break;

        case 1:
            cdv = 1;
            hc_c_size = ext_bits(cs_sdd, 63, 48);
            blocks = (hc_c_size+1)*1024;
            debug_if(SD_DBG, "\n\rSDHC Card \n\rhc_c_size: %d\n\rcapacity: %lld \n\rsectors: %lld\n\r", hc_c_size, blocks*512, blocks);
            break;

        default:
            debug("cs_sdD struct unsupported\r\n");
            return 0;
    };
    return blocks;
}

int cmdx(int cmd, int arg)
{
    cs_sd = 0;

    // send a command
    spi_sd.write(0x40 | cmd);
    spi_sd.write(arg >> 24);
    spi_sd.write(arg >> 16);
    spi_sd.write(arg >> 8);
    spi_sd.write(arg >> 0);
    spi_sd.write(0x95);

    // wait for the repsonse (response[7] == 0)
    for (int i = 0; i < SD_COMMAND_TIMEOUT; i++) {
        int response = spi_sd.write(0xFF);
        if (!(response & 0x80)) {
            return response;
        }
    }
    cs_sd = 1;
    spi_sd.write(0xFF);
    return -1; // timeout
}

static uint32_t ext_bits(unsigned char *data, int msb, int lsb)
{
    uint32_t bits = 0;
    uint32_t size = 1 + msb - lsb;
    for (int i = 0; i < size; i++) {
        uint32_t position = lsb + i;
        uint32_t byte = 15 - (position >> 3);
        uint32_t bit = position & 0x7;
        uint32_t value = (data[byte] >> bit) & 1;
        bits |= value << i;
    }
    return bits;
}

int disk_initialize()
{
    int i = initialise_card();
    debug_if(SD_DBG, "init card = %d\n", i);
    sectors = sd_sectors();

    // Set block length to 512 (CMD16)
    if (cmd(16, 512) != 0) {
        debug("\rSet 512-byte block timed out\r\n");
        return 1;
    } else {
        printf("\rDisk initialization successfull\r\n");
    }

    spi_sd.frequency(1000000); // Set to 1MHz for data transfer
    return 0;
}

int disk_write(const uint8_t *buffer, uint64_t block_number)

{
    // set write address for single block (CMD24)
    if (cmd(24, block_number * cdv) != 0) {
        return 1;
    }

    // send the data block
    write(buffer, 512);
    //printf("Written Successfully bro \n");
    return 0;
}

int write(const uint8_t*buffer, uint32_t length)
{
    cs_sd = 0;

    // indicate start of block
    spi_sd.write(0xFE);

    // write the data
    for (int i = 0; i < length; i++) {
        spi_sd.write(buffer[i]);
    }

    // write the checksum
    spi_sd.write(0xFF);
    spi_sd.write(0xFF);

    // check the response token
    if ((spi_sd.write(0xFF) & 0x1F) != 0x05) {
        cs_sd = 1;
        spi_sd.write(0xFF);
        return 1;
    }

    // wait for write to finish
    while (spi_sd.write(0xFF) == 0);

    cs_sd = 1;
    spi_sd.write(0xFF);
    return 0;
}

int disk_read(uint8_t *buffer, uint64_t block_number)
{
    // set read address for single block (CMD17)
    if (cmd(17, block_number * cdv) != 0) {
        return 1;
    }

    // receive the data
    read(buffer, 512);
    return 0;
}

int read(uint8_t *buffer, uint32_t length)
{
    cs_sd = 0;

    // read until start byte (0xFF)
    while (spi_sd.write(0xFF) != 0xFE);

    // read data
    for (int i = 0; i < length; i++) {
        buffer[i] = spi_sd.write(0xFF);
    }
    spi_sd.write(0xFF); // checksum
    spi_sd.write(0xFF);

    cs_sd = 1;
    spi_sd.write(0xFF);
    return 0;
}

int disk_erase(int startBlock, int totalBlocks)
{
    if(cmd(32, startBlock * cdv) != 0) {
        return 1;
    }
    if (cmd(33, (startBlock+totalBlocks-1) * cdv) != 0) {
        return 1;
    }
    if (cmd(38,0) != 0) {
        return 1;
    }
    
    return 0; //normal return
}
