#include "mbed.h"
#include "main.h"
#include "sx1276-hal.h"
#include "debug.h"
#include "ordre.h"
#include "trame.h"

/* Set this flag to '1' to display debug messages on the console 
    Must be 0 in order to work with the projet
*/
#define DEBUG_MESSAGE   0

#define ID_DEVICE 0x24
#define ID_STATION 0x55
#define TEMPS_ATTENTE_COM 20

/* Set this flag to '1' to use the LoRa modulation or to '0' to use FSK modulation */
#define USE_MODEM_LORA  1
#define USE_MODEM_FSK   !USE_MODEM_LORA

#define RF_FREQUENCY                                    865000000 // Hz
#define TX_OUTPUT_POWER                                 14        // 14 dBm

#if USE_MODEM_LORA == 1

    #define LORA_BANDWIDTH                              2         // [0: 125 kHz,
                                                                  //  1: 250 kHz,
                                                                  //  2: 500 kHz,
                                                                  //  3: Reserved]
    #define LORA_SPREADING_FACTOR                       7         // [SF7..SF12]
    #define LORA_CODINGRATE                             1         // [1: 4/5,
                                                                  //  2: 4/6,
                                                                  //  3: 4/7,
                                                                  //  4: 4/8]
    #define LORA_PREAMBLE_LENGTH                        8         // Same for Tx and Rx
    #define LORA_SYMBOL_TIMEOUT                         5         // Symbols
    #define LORA_FIX_LENGTH_PAYLOAD_ON                  false
    #define LORA_FHSS_ENABLED                           false  
    #define LORA_NB_SYMB_HOP                            4     
    #define LORA_IQ_INVERSION_ON                        false
    #define LORA_CRC_ENABLED                            true
    
#elif USE_MODEM_FSK == 1

    #define FSK_FDEV                                    25000     // Hz
    #define FSK_DATARATE                                19200     // bps
    #define FSK_BANDWIDTH                               50000     // Hz
    #define FSK_AFC_BANDWIDTH                           83333     // Hz
    #define FSK_PREAMBLE_LENGTH                         5         // Same for Tx and Rx
    #define FSK_FIX_LENGTH_PAYLOAD_ON                   false
    #define FSK_CRC_ENABLED                             true
    
#else
    #error "Please define a modem in the compiler options."
#endif

#define RX_TIMEOUT_VALUE                                3500000   // in us
#define BUFFER_SIZE                                     32        // Define the payload size here

#if( defined ( TARGET_KL25Z ) || defined ( TARGET_LPC11U6X ) )
DigitalOut led(LED2);
#else
DigitalOut led(LED1);
#endif

/*
 *  Global variables declarations
 */
typedef enum
{
    LOWPOWER = 0,
    IDLE,
    
    RX,
    RX_TIMEOUT,
    RX_ERROR,
    
    TX,
    TX_TIMEOUT,
    
    CAD,
    CAD_DONE
}AppStates_t;

volatile AppStates_t State = LOWPOWER;

/*!
 * Radio events function pointer
 */
static RadioEvents_t RadioEvents;

/*
 *  Global variables declarations
 */
SX1276MB1xAS Radio( NULL );

uint16_t BufferSize = BUFFER_SIZE;
uint8_t Buffer[BUFFER_SIZE];

int16_t RssiValue = 0.0;
int8_t SnrValue = 0.0;

//Connection serie avec le PC
Serial pc(USBTX, USBRX);

char portCom[3];
bool ordreRecuPortCom = false;


int main() 
{
    uint8_t i;
    bool isMaster = false;
    
    debug_if( DEBUG_MESSAGE,  "\n\n\r     SX1276 Server Demo Application \n\n\r" );

    // Initialize Radio driver
    RadioEvents.TxDone = OnTxDone;
    RadioEvents.RxDone = OnRxDone;
    RadioEvents.RxError = OnRxError;
    RadioEvents.TxTimeout = OnTxTimeout;
    RadioEvents.RxTimeout = OnRxTimeout;
    Radio.Init( &RadioEvents );
    
    // verify the connection with the board
    while( Radio.Read( REG_VERSION ) == 0x00  )
    {
        debug_if( DEBUG_MESSAGE,  "Radio could not be detected!\n\r", NULL );
        wait( 1 );
    }
            
    debug_if( ( DEBUG_MESSAGE & ( Radio.DetectBoardType( ) == SX1276MB1LAS ) ) , "\n\r > Board Type: SX1276MB1LAS < \n\r" );
    debug_if( ( DEBUG_MESSAGE & ( Radio.DetectBoardType( ) == SX1276MB1MAS ) ) , "\n\r > Board Type: SX1276MB1MAS < \n\r" );
    
    Radio.SetChannel( RF_FREQUENCY ); 

#if USE_MODEM_LORA == 1
    
    debug_if( DEBUG_MESSAGE & LORA_FHSS_ENABLED, "\n\n\r             > LORA FHSS Mode < \n\n\r");
    debug_if( DEBUG_MESSAGE & !LORA_FHSS_ENABLED, "\n\n\r             > LORA Mode < \n\n\r");

    Radio.SetTxConfig( MODEM_LORA, TX_OUTPUT_POWER, 0, LORA_BANDWIDTH,
                         LORA_SPREADING_FACTOR, LORA_CODINGRATE,
                         LORA_PREAMBLE_LENGTH, LORA_FIX_LENGTH_PAYLOAD_ON,
                         LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP, 
                         LORA_IQ_INVERSION_ON, 2000000 );
    
    Radio.SetRxConfig( MODEM_LORA, LORA_BANDWIDTH, LORA_SPREADING_FACTOR,
                         LORA_CODINGRATE, 0, LORA_PREAMBLE_LENGTH,
                         LORA_SYMBOL_TIMEOUT, LORA_FIX_LENGTH_PAYLOAD_ON, 0,
                         LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP, 
                         LORA_IQ_INVERSION_ON, true );
                         
#elif USE_MODEM_FSK == 1

    debug_if( DEBUG_MESSAGE, "\n\n\r              > FSK Mode < \n\n\r");
    Radio.SetTxConfig( MODEM_FSK, TX_OUTPUT_POWER, FSK_FDEV, 0,
                         FSK_DATARATE, 0,
                         FSK_PREAMBLE_LENGTH, FSK_FIX_LENGTH_PAYLOAD_ON,
                         FSK_CRC_ENABLED, 0, 0, 0, 2000000 );
    
    Radio.SetRxConfig( MODEM_FSK, FSK_BANDWIDTH, FSK_DATARATE,
                         0, FSK_AFC_BANDWIDTH, FSK_PREAMBLE_LENGTH,
                         0, FSK_FIX_LENGTH_PAYLOAD_ON, 0, FSK_CRC_ENABLED,
                         0, 0, false, true );
                         
#else

#error "Please define a modem in the compiler options."

#endif
     
    debug_if( DEBUG_MESSAGE, "Starting Server loop\r\n" ); 
        
    led = 0;
    
    //ACK Message Par defaut
    char  AckMsg[] = {0x24, 0x55, 0x00};
    
    Radio.Rx( RX_TIMEOUT_VALUE );
    
    //Varialbe de controle
    bool ackOrData = true; //true = ack, false = Data
    
    while( 1 )
    {
        switch( State )
        {
        case RX:
            if( BufferSize > 0 )
            {
                TrameData trameRecue((char*)Buffer);
                if( trameRecue.getIdStation() == (char) ID_STATION )
                {
                    
                    led = !led;
                    debug_if( DEBUG_MESSAGE,  "...DATA\r\n" );
                    
                    //Envoit sur le port COM
                    pc.printf(trameRecue.creerTrame());
                    
                    // Send the ACK
                    AckMsg[0] = trameRecue.getIdEmetteur();
                    
                    strcpy( ( char* )Buffer, ( char* )AckMsg );
                    // We fill the buffer with numbers for the payload 
                    for( i = 3; i < BufferSize; i++ )
                    {
                        Buffer[i] = i - 4;
                    }
                    wait( 5.0);
                    ackOrData = true;
                    Radio.Send( Buffer, BufferSize );
                    
                }
                else // valid reception but not expected
                {    
                    Radio.Rx( RX_TIMEOUT_VALUE );
                }    
            }
            State = LOWPOWER;
            break;
        case TX:    
            led = !led; 
            if(ackOrData){
                debug_if( DEBUG_MESSAGE,  "ACK...\r\n" );
                //Après Envoie du ACK, on ecoute la réponse du PC (pour l'ordre)
                led = 1;
                pc.scanf("%s",portCom);
                led = 0;
                //Mise a jour des variables de controle
                ordreRecuPortCom = true;
                ackOrData = false;
            }
            else{
                debug_if( DEBUG_MESSAGE,  "ORDRE...\r\n" );
                Radio.Rx( RX_TIMEOUT_VALUE );
            }
            State = LOWPOWER;
            break;
        case RX_TIMEOUT:
            Radio.Rx( RX_TIMEOUT_VALUE );             
            State = LOWPOWER;
            break;
        case RX_ERROR:
            // We have received a Packet with a CRC error, send reply as if packet was correct
            //Do nothing
            State = LOWPOWER;
            break;
        case TX_TIMEOUT:
            Radio.Rx( RX_TIMEOUT_VALUE );
            State = LOWPOWER;
            break;
        case LOWPOWER:
            if(ordreRecuPortCom){
                Ordre o((char *)portCom);
                debug_if( DEBUG_MESSAGE,  "Ordre recu sur le port COM\r\n" );
                led=!led;
                if(o.getOrdreAFaire() == 1){
                    //Envoie de l'ordre.
                    strcpy( ( char* )Buffer, ( char* )portCom );
                    for( i = 3; i < BufferSize; i++ )
                    {
                        Buffer[i] = i - 4;
                    }
                    wait_ms( 10 );
                    ordreRecuPortCom = false;
                    Radio.Send( Buffer, BufferSize );
                }else{
                    debug_if( DEBUG_MESSAGE,  "Pas d'activation de la pompe\r\n" );
                    Radio.Rx( RX_TIMEOUT_VALUE );
                }
                ordreRecuPortCom = false;
            
            break;
        default:
            State = LOWPOWER;
            break;
        }    
    }
}

void OnTxDone( void )
{
    Radio.Sleep( );
    State = TX;
    debug_if( DEBUG_MESSAGE, "> OnTxDone\n\r" );
}

void OnRxDone( uint8_t *payload, uint16_t size, int16_t rssi, int8_t snr)
{
    Radio.Sleep( );
    BufferSize = size;
    memcpy( Buffer, payload, BufferSize );
    RssiValue = rssi;
    SnrValue = snr;
    State = RX;
    debug_if( DEBUG_MESSAGE, "> OnRxDone\n\r" );
}

void OnTxTimeout( void )
{
    Radio.Sleep( );
    State = TX_TIMEOUT;
    debug_if( DEBUG_MESSAGE, "> OnTxTimeout\n\r" );
}

void OnRxTimeout( void )
{
    Radio.Sleep( );
    Buffer[ BufferSize ] = 0;
    State = RX_TIMEOUT;
    debug_if( DEBUG_MESSAGE, "> OnRxTimeout\n\r" );
}

void OnRxError( void )
{
    Radio.Sleep( );
    State = RX_ERROR;
    debug_if( DEBUG_MESSAGE, "> OnRxError\n\r" );
}

