#include "mbed.h"
#include "main.h"
#include "sx1272-hal.h"
#include "debug.h"
#include "trame.h"
#include "ordre.h"
#include "pompe.h"
#include "WakeUp.h"
#include "niveau.h"

/* Set this flag to '1' to display debug messages on the console */
#define DEBUG_MESSAGE   1
#define ID_DEVICE 0x24
#define ID_STATION 0x55
#define TEMPS_ECOUTE_ORDRE_SECONDE 20.0
#define TEMPS_ATTENTE_ACK 60.0
#define SLEEP_TIME 30000 //ms

/* Set this flag to '1' to use the LoRa modulation or to '0' to use FSK modulation */
#define USE_MODEM_LORA  1
#define USE_MODEM_FSK   !USE_MODEM_LORA

#define RF_FREQUENCY                                    865000000 // Hz
#define TX_OUTPUT_POWER                                 14        // 14 dBm

#if USE_MODEM_LORA == 1

    #define LORA_BANDWIDTH                              2         // [0: 125 kHz,
                                                                  //  1: 250 kHz,
                                                                  //  2: 500 kHz,
                                                                  //  3: Reserved]
    #define LORA_SPREADING_FACTOR                       7         // [SF7..SF12]
    #define LORA_CODINGRATE                             1         // [1: 4/5,
                                                                  //  2: 4/6,
                                                                  //  3: 4/7,
                                                                  //  4: 4/8]
    #define LORA_PREAMBLE_LENGTH                        8         // Same for Tx and Rx
    #define LORA_SYMBOL_TIMEOUT                         5         // Symbols
    #define LORA_FIX_LENGTH_PAYLOAD_ON                  false
    #define LORA_FHSS_ENABLED                           false  
    #define LORA_NB_SYMB_HOP                            4     
    #define LORA_IQ_INVERSION_ON                        false
    #define LORA_CRC_ENABLED                            true
    
#elif USE_MODEM_FSK == 1

    #define FSK_FDEV                                    25000     // Hz
    #define FSK_DATARATE                                19200     // bps
    #define FSK_BANDWIDTH                               50000     // Hz
    #define FSK_AFC_BANDWIDTH                           83333     // Hz
    #define FSK_PREAMBLE_LENGTH                         5         // Same for Tx and Rx
    #define FSK_FIX_LENGTH_PAYLOAD_ON                   false
    #define FSK_CRC_ENABLED                             true
    
#else
    #error "Please define a modem in the compiler options."
#endif

#define RX_TIMEOUT_VALUE                                3500000//3500000   // in us
#define BUFFER_SIZE                                     32        // Define the payload size here

#if( defined ( TARGET_KL25Z ) || defined ( TARGET_LPC11U6X ) )
DigitalOut led(LED2);
#else
DigitalOut led(LED1);
#endif

/*
 *  Global variables declarations
 */
typedef enum
{
    LOWPOWER = 0,
    IDLE,
    
    RX,
    RX_TIMEOUT,
    RX_ERROR,
    
    TX,
    TX_TIMEOUT,
    
    CAD,
    CAD_DONE
}AppStates_t;

volatile AppStates_t State = LOWPOWER;

/*!
 * Radio events function pointer
 */
static RadioEvents_t RadioEvents;

/*
 *  Global variables declarations
 */
SX1272MB2xAS Radio( NULL );

uint16_t BufferSize = BUFFER_SIZE;
uint8_t Buffer[BUFFER_SIZE];

int16_t RssiValue = 0.0;
int8_t SnrValue = 0.0;

//Fonction permettant de faire dormir la carte
//LE DEEPSLEEP Ne fonctionne pas !
void dormir(){
    debug_if(DEBUG_MESSAGE,"dodo\r\n");
    Radio.Sleep();
    WakeUp::set_ms(SLEEP_TIME);
    sleep();
    
    wait(5);
}

Timer ecouterOrdre;

int main() 
{
    uint8_t i;
    
    debug_if(DEBUG_MESSAGE, "\n\n\r     SX1272 Cuve Demo Application \n\n\r" );

    // Initialize Radio driver
    RadioEvents.TxDone = OnTxDone;
    RadioEvents.RxDone = OnRxDone;
    RadioEvents.RxError = OnRxError;
    RadioEvents.TxTimeout = OnTxTimeout;
    RadioEvents.RxTimeout = OnRxTimeout;
    Radio.Init( &RadioEvents );
    
    // verify the connection with the board
    while( Radio.Read( REG_VERSION ) == 0x00  )
    {
        debug_if(DEBUG_MESSAGE, "Radio could not be detected!\n\r", NULL );
        wait( 1 );
    }
    
    debug_if( ( DEBUG_MESSAGE & ( Radio.DetectBoardType( ) == SX1272MB2XAS ) ) , "\n\r > Board Type: SX1272MB2xAS < \n\r" );
  
    Radio.SetChannel( RF_FREQUENCY ); 

#if USE_MODEM_LORA == 1
    
    debug_if( LORA_FHSS_ENABLED, "\n\n\r             > LORA FHSS Mode < \n\n\r");
    debug_if( !LORA_FHSS_ENABLED, "\n\n\r             > LORA Mode < \n\n\r");

    Radio.SetTxConfig( MODEM_LORA, TX_OUTPUT_POWER, 0, LORA_BANDWIDTH,
                         LORA_SPREADING_FACTOR, LORA_CODINGRATE,
                         LORA_PREAMBLE_LENGTH, LORA_FIX_LENGTH_PAYLOAD_ON,
                         LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP, 
                         LORA_IQ_INVERSION_ON, 2000000 );
    
    Radio.SetRxConfig( MODEM_LORA, LORA_BANDWIDTH, LORA_SPREADING_FACTOR,
                         LORA_CODINGRATE, 0, LORA_PREAMBLE_LENGTH,
                         LORA_SYMBOL_TIMEOUT, LORA_FIX_LENGTH_PAYLOAD_ON, 0,
                         LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP, 
                         LORA_IQ_INVERSION_ON, true );
                         
#elif USE_MODEM_FSK == 1

    debug_if(DEBUG_MESSAGE,"\n\n\r              > FSK Mode < \n\n\r");
    Radio.SetTxConfig( MODEM_FSK, TX_OUTPUT_POWER, FSK_FDEV, 0,
                         FSK_DATARATE, 0,
                         FSK_PREAMBLE_LENGTH, FSK_FIX_LENGTH_PAYLOAD_ON,
                         FSK_CRC_ENABLED, 0, 0, 0, 2000000 );
    
    Radio.SetRxConfig( MODEM_FSK, FSK_BANDWIDTH, FSK_DATARATE,
                         0, FSK_AFC_BANDWIDTH, FSK_PREAMBLE_LENGTH,
                         0, FSK_FIX_LENGTH_PAYLOAD_ON, 0, FSK_CRC_ENABLED,
                         0, 0, false, true );
                         
#else

#error "Please define a modem in the compiler options."

#endif
     
    debug_if( DEBUG_MESSAGE, "Starting Cuve loop\r\n" ); 
        
    led = 0;
    
    //Initialisation
    Pompe pompe(PC_5);
    pompe.arreterPompe();
    Niveau cuve(PC_8,PC_3,PC_2,PC_6);
    TrameData dataPaquet(ID_DEVICE,ID_STATION, 8, pompe.etat(), 0xF, 0xA);
    
    bool enAttenteOrdre = false;
    bool enAttenteAck = false;
    
    //The low-power oscillator can be quite inaccurate on some targets
    //this function calibrates it against the main clock
    WakeUp::calibrate();
    
    
    //Debut programme
    while( 1 )
    {
        switch( State )
        {
        case RX:
            if( enAttenteOrdre || enAttenteAck )
            {
               
                if( BufferSize > 0 )
                {
                    debug_if( DEBUG_MESSAGE, "2\r\n");
                    debug_if(DEBUG_MESSAGE,(const char* )Buffer);
                    debug_if( DEBUG_MESSAGE, " ");
                    debug_if(DEBUG_MESSAGE,(const char* ) ID_DEVICE);
                    Ordre trameRecue((char*)Buffer);
                    //Si message recu alors trouver ordre correspondant et faire traitement
                    if( trameRecue.getIdRecepteur() == (char) ID_DEVICE )
                    {
                        debug_if( DEBUG_MESSAGE, "ID DEVICE OK\r\n");
                        if( trameRecue.getIdEmetteur() == (char) ID_STATION ){
                            debug_if( DEBUG_MESSAGE, "ID STATION OK\r\n");
                            
                            if(enAttenteOrdre){
                                //Ordre pour nous
                                led = !led;
                                debug_if( DEBUG_MESSAGE,  "...Ordre recu \r\n" );
                                Ordre o((char *) Buffer);
                                if(trameRecue.getOrdreAFaire()==1)
                                    o.executerOrdre(pompe, cuve, o.getNiveauCuve());
                                
                                dormir();
                                enAttenteOrdre = false;
                                Radio.Rx( RX_TIMEOUT_VALUE );
                            }
                            else if(trameRecue.getOrdreAFaire()==0 && enAttenteAck){
                                debug_if( DEBUG_MESSAGE, "ACK recu\r\n");
                                enAttenteAck = false;
                                ecouterOrdre.reset();
                                enAttenteOrdre = true;
                                Radio.Rx( RX_TIMEOUT_VALUE );
                            }
                            else{
                                //mauvais ACK on concidère que c'est OK
                                debug_if( DEBUG_MESSAGE, "mauvais ACK recu\r\n");
                                enAttenteAck = false;
                                ecouterOrdre.reset();
                                enAttenteOrdre = true;
                                Radio.Rx( RX_TIMEOUT_VALUE );
                            }
                        }else{
                            debug_if( DEBUG_MESSAGE, "autre station\r\n");
                            Radio.Rx( RX_TIMEOUT_VALUE );
                        }
                    }
                    else // valid reception but not for us
                    {    // Start again
                        debug_if( DEBUG_MESSAGE, "Reception message pour quelqu'un d'autre\r\n");
                        Radio.Rx( RX_TIMEOUT_VALUE );
                    }    
                }
            }
            State = LOWPOWER;
            break;
        case TX:    
            led = !led; 
            debug_if( DEBUG_MESSAGE && !enAttenteAck,  "Envoi des donnees...\r\n" );
            debug_if( DEBUG_MESSAGE && enAttenteAck, "En attente ACK\r\n");
            Radio.Rx( RX_TIMEOUT_VALUE );
            State = LOWPOWER;
            break;
        case RX_TIMEOUT:           
            Radio.Rx( RX_TIMEOUT_VALUE );  
            State = LOWPOWER;
            break;
        case RX_ERROR:
            //Erreur CRC, faire une demande de renvoi
            //TODO
            debug_if( DEBUG_MESSAGE, "Erreur CRC \r\n");
            Radio.Rx( RX_TIMEOUT_VALUE );
            State = LOWPOWER;
            break;
        case TX_TIMEOUT:
            Radio.Rx( RX_TIMEOUT_VALUE );
            State = LOWPOWER;
            break;
        case LOWPOWER:
            if(!enAttenteOrdre && !enAttenteAck){
                //Corps de l'application
                //On envoie les données
                debug_if( DEBUG_MESSAGE, "envoie3\r\n");
                dataPaquet.mettreAJourEtatPompe((char) pompe.etat());
                dataPaquet.mettreAJourNiveauCuve(cuve.getNiveauCuve());
                dataPaquet.mettreAJourNiveauBatterie((char) 0xA);
                
                // Send the data
                char * trame = dataPaquet.creerTrame();
                strcpy( ( char* ) Buffer, trame);
                for (i = TAILLE_TRAME_DATA; i < BufferSize; i++ )
                {
                    Buffer[i] = i - 4;
                }
                wait_ms( 10 );
                
                //Initialisation des variables de controle
                enAttenteAck = true;
                debug_if( DEBUG_MESSAGE,  "Debut timer \r\n");
                ecouterOrdre.start();
                Radio.Send( Buffer, BufferSize );
            }else{
                ecouterOrdre.stop();
                if(enAttenteAck && (ecouterOrdre.read() > TEMPS_ATTENTE_ACK)){
                    debug_if( DEBUG_MESSAGE, "delaisAttente ack depasse. Reemission.\r\n");
                    ecouterOrdre.stop();
                    ecouterOrdre.reset();
                    enAttenteAck = false;
                }else
                if(enAttenteOrdre && (ecouterOrdre.read() > TEMPS_ECOUTE_ORDRE_SECONDE)){
                    debug_if( DEBUG_MESSAGE, "delaisAttente depasse. Dormir.\r\n");
                    ecouterOrdre.stop();
                    ecouterOrdre.reset();
                    dormir();
                    debug_if( DEBUG_MESSAGE, "fin dodo\r\n");
                    enAttenteOrdre = false;
                }
                else ecouterOrdre.start();
            }
            break;
        default:
            State = LOWPOWER;
            break;
        }    
    }
}

void OnTxDone( void )
{
    Radio.Sleep( );
    State = TX;
    debug_if( DEBUG_MESSAGE, "> OnTxDone\n\r" );
}

void OnRxDone( uint8_t *payload, uint16_t size, int16_t rssi, int8_t snr)
{
    Radio.Sleep( );
    BufferSize = size;
    memcpy( Buffer, payload, BufferSize );
    RssiValue = rssi;
    SnrValue = snr;
    State = RX;
    debug_if( DEBUG_MESSAGE, "> OnRxDone\n\r" );
}

void OnTxTimeout( void )
{
    Radio.Sleep( );
    State = TX_TIMEOUT;
    debug_if( DEBUG_MESSAGE, "> OnTxTimeout\n\r" );
}

void OnRxTimeout( void )
{
    Radio.Sleep( );
    Buffer[ BufferSize ] = 0;
    State = RX_TIMEOUT;
    debug_if( DEBUG_MESSAGE, "." );
}

void OnRxError( void )
{
    Radio.Sleep( );
    State = RX_ERROR;
    debug_if( DEBUG_MESSAGE, "> OnRxError\n\r" );
}

