#include "Glider.h"

Glider::Glider(Serial* device, PinName sda, PinName scl, PinName tx, PinName rx) : xbeeRxInt(rx), pitot(p15), solVoltIn(p16), buzzer(p26) {
    dev = device;
    dev->printf("constructing glider...");
    initSaveData();
    xbee = new Serial(tx, rx);
    heading = 0.0;
    pressure = 0.0;
    temp = 0.0;
    alt = 0.0;
    cmdFlag = false;
    transmitFlag = false;
    hmc = new HMC5883L(sda, scl);
    hmc->init();
    bmp = new BMP180(sda, scl);
    bmp->Initialize(64, BMP180_OSS_ULTRA_LOW_POWER);
    xbeeRxInt.rise(this, &Glider::setCommandFlag);
}

void Glider::setHeading() {
    this->heading = this->hmc->getHeading();
}

void Glider::setTempPressAlt() {
    bmp->ReadData(&this->temp, &this->pressure, &this->alt);
}

void Glider::setAirSpeed() {
    float vInput =  this->pitot.read_u16() * PITO_ADC_RATIO;
    vInput = vInput / PITO_V_DIVIDER;
    
    float diffPressure = (vInput - 0.5) / (0.2 * 5.0);  //kPa
    this->speed = sqrt((2*diffPressure) / AIR_DENSITY);
}

void Glider::setSolarVoltage() {
    solarVoltage = solVoltIn.read_u16() * PITO_ADC_RATIO;
    solarVoltage = solarVoltage / SOLAR_V_DIVIDER;   
}

void Glider::setMissionTime() {
    this->missionTime = time(NULL) - this->startTime;  //duration = current tome - start time
}

void Glider::updateTelem() {
    setHeading();
    setTempPressAlt();
    setAirSpeed();
    setSolarVoltage();
    setMissionTime();
}

void Glider::setTransmitFlag() {
    this->transmitFlag = true;   
}

void Glider::forceLandState() {
    state = LAND;
}

void Glider::saveTelem() {
     FILE *fp = fopen("/telem/telem.csv", "a");
     fprintf(fp, PACKET_FORMAT, 
                        this->missionTime,  //duration in seconds 
                        this->packetCount,  
                        this->alt, //meters
                        this->pressure, //
                        this->temp,  //C
                        this->speed,  //m per s
                        this->solarVoltage,
                        this->heading, //degrees from North
                        this->state);
     fclose(fp);
}

void Glider::initSaveData() {
     this->dev->printf("getting the time...\r\n");
     FILE *fp = fopen("/telem/data.txt", "r");   //attempt to read the saved data
     
     if (fp == NULL) {  //file does not exist
        this->dev->printf("starting the time...\r\n");
        this->startTime = time(NULL); //initialize the start time to now
        this->packetCount = 0;  //start the packet count at 0
        this->state = CRUZE;
        FILE *fp1 = fopen("/telem/data.txt", "w");  //create the data file
        fprintf(fp1, SAVE_DATA_FORMAT, this->startTime, this->packetCount, this->state);  //save the start time
        fclose(fp1);
        return;   
     }
     
     //else, the file exists.  Read the data
     fscanf(fp, SAVE_DATA_FORMAT, &this->startTime, &this->packetCount, &this->state);  //set the start time, packet count, and state
     rewind(fp);  //move cursor back to beginning of file
     fclose(fp);
}

void Glider::saveData() { 
     FILE *fp = fopen("/telem/data.txt", "w");
     fprintf(fp, SAVE_DATA_FORMAT, this->startTime, this->packetCount, this->state);  //save the start time, packet count, and state to flash
     fclose(fp);   
}

void Glider::transmitPacket() {
    this->packetCount++;
    
    this->xbee->printf(PACKET_FORMAT, 
                        this->missionTime,  //duration in seconds 
                        this->packetCount,  
                        this->alt, //meters
                        this->pressure, //
                        this->temp,  //C
                        this->speed,  //m per s
                        this->solarVoltage,
                        this->heading, //degrees from North
                        this->state);
}

void Glider::setCommandFlag() {
    this->cmdFlag = true;   
}

void Glider::processCommand() {
    if (this->cmdFlag) {  //command recieved
        char command = this->xbee->getc();  //get the command char
        
        switch (command) {
            case CMD_BUZZER:  //'b' was recieved.  sound the buzzer
                  this->state = LAND; //set state to land
                  break; 
            case CMD_RESET:  //'z' was recieved
                  xbee->printf("resetting save file...\r\n");
                  
                  state = 0;
                  packetCount = 0;
                  startTime = 0;
                  saveData();
                  
                  
                  //remove("/telem/data.txt");
                  
                  break;
                  
            case CMD_NEXT_STATE:
                  if (state < 1 && state >= 0) {
                      xbee->printf("Going to next state...\r\n");
                      state++;
                  }
        }
        
        this->cmdFlag = false;  //reset the flag so a new command can be recieved
    }  
}

void Glider::soundBuzzer(float freq, float dur, bool infinate) {
    buzzer.beep(freq, dur, infinate);    
}

void Glider::checkForLand() {
    xbee->printf("%f\r\n", alt);
    if (alt <= ALT_THRESH) {
        state = LAND;   
    }   
}

void Glider::endMission() {
    this->transmitPacket();  //transmit one last packet
    this->soundBuzzer(BUZZER_FREQ, 1, true);  //sound buzzer forever
    while (1) {
         processCommand();   
    }
}
