#ifndef rfm12B_H
#define rfm12B_H

#include <mbed.h>

/*!
 * \file       rfm12b.h
 * \brief      class for rfm2b in rawmode - only receive part implemented
 * \author     Karl Zweimüller based on code from WED 6.9.2009
 */

typedef unsigned char Byte;    // used to be uint8_t : something a byte wide, whatever ....

/** This Class handles a rfm12b transceiver
 * see http://www.hoperf.com/rf_fsk/rfm12b.htm
 *
*/
class rfm12b {
public:
    /** Create a rfm12b object
    *
    * @param mosi SPI-Interface. One of the 2 PSI-Interfaces of mbed. Pin p5 or p11
    * @param miso SPI-Interface. One of the 2 PSI-Interfaces of mbed. Pin p6 or p12
    * @param sclk SPI-Interface. One of the 2 PSI-Interfaces of mbed. Pin p7 or p13
    * @param nsel Chip-Select.   A Digial Output of mbed
    * @param rxdata  Data-Pin for received data. A DigitalIn of mbed
    */
    rfm12b(PinName mosi, PinName miso, PinName sclk, PinName nsel, PinName rxdata);

    /** init the spi-interface
    */
    void init_spi();

    /** initialize the device
    */
    void RFM_init(void);

    /** write and read 16 bit
    */
    uint16_t rfm_spi16(uint16_t outval);

    /** attach a function to be called when the data-pin changes from 0->1 and from 1->0
    * this function has to do all the decoding
    * keep this function short, as no interrrupts can occour within
    *
    * @param fptr Pointer to callback-function
    */
    void attachISR(void (*fptr)(void)) {
        m_pinRXData->fall(fptr);
        m_pinRXData->rise(fptr);
    }

    template<typename T>
    /** attach an object member function to be called when the data-pin changes from 0->1 and from 1->0
    *
    * @param tptr pointer to object
    * @param mprt pointer ro member function
    *
    */
    void attachISR(T* tptr, void (T::*mptr)(void)) {
        if ((mptr != NULL) && (tptr != NULL)) {
            m_pinRXData->fall(tptr, mptr);
            m_pinRXData->rise(tptr, mptr);
        }
    }



private:


    DigitalOut    *cs;          //chipselect
    InterruptIn   *m_pinRXData; //rx data pin
    SPI           *rfm12b_spi;  //spi-interface


};

#endif
