/** mbed Library for scrolling text in Text-LCDs based on the Class TextLCD
 * Copyright by Karl Zweimueller
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
*/

#ifndef TEXTLCDSCROLL_H
#define TEXTLCDSCROLL_H

#include "TextLCD.h"


/** TextLCDScroll class which handles horizontal scrolling text if it doesn't fit in one line, based on class TextLCD
  *
  * Example:
  * @code
  * #include "TextLCDScroll.h"
  * #include "mbed.h"
  *
  * TextLCDScroll lcd(p30, p29, p28, p27, p26, p25, TextLCD::LCD16x2); // rs, e, d0-d3
  *
  * int main() {
  *
  *     lcd.cls();
  *     lcd.setLine(0,"TextLCDScroll!");
  *     lcd.setLine(1,"This line can be rather long. The quick brown fox jumps over the lazy dog.");
  * }
  * @endcode
*/
class TextLCDScroll: public TextLCD {
public:

    /** How should we scroll horizontal in one line
      */
    enum ScrollModes {
        leftright      /** Scroll left ande the right */
        ,left          /** Scroll only left - default */
    };

    /** Create a TextLCDScroll Object with pins and Type from TextLCD
      *
      * @param rs    Instruction/data control line
      * @param e     Enable line (clock)
      * @param d4-d7 Data lines for using as a 4-bit interface
      * @param type  Sets the panel size/addressing mode (default = LCD16x2)
      */
    TextLCDScroll(PinName rs, PinName e, PinName d4, PinName d5,
                  PinName d6, PinName d7, TextLCD::LCDType type);


    /** Display a string str on line Line. Replaces old text on this Line.
      * @param  Line    Line, where text should be displayed. first Line=0
      * @param  *str    the string to display (c-string)
      */
    bool setLine( int Line, char *str);
    
    /* Clear the display
    *
    */
    void cls();

    /** Set the speed, how fast to scroll the text in Characters per Second
     * @param  speed Speed for scrolling in Characters per Second. Range 0.1 ... 10
     */
    bool setSpeed( int speed);

    /** Set the Scroll-Mode. Deault "left"
      * @param  mode    a valid mode. See ScrollModes
      */
    bool setScrollMode( ScrollModes mode);

private:
    void startTicker();

    void ScrollRightLeft();

    void ScrollLeft();

    Ticker tick;
    ScrollModes _mode;

    // these are changed in interrupt-routine!
    volatile int* _direction;
    volatile int* _actPos;
    volatile int _speed;
    
    char* line[99];

};

#endif