#ifndef NAVIGATOR_H
#define NAVIGATOR_H

#include "mbed.h"
#include "Menu.h"
#include "TextLCD.h"

/** Class Navigator does the Navigation in the menu and updates the display.
 *  Interaction from outside is done by calling moveUp(), moveDown() or select().
 *  Could be done by an RPG or via Buttons. 
 *
 * Example:
 *
 * @code
 *
 * #include "MenuItem.h"
 * #include "Menu.h"
 * #include "Navigator.h"
 * #include <vector>
 * #include <string>
 * PinDetect  T1 ( p21,PullNone); //Button 1 - UP
 * PinDetect  T2 ( p22,PullNone); //Button 2 - Down
 * PinDetect  T3 ( p23,PullNone); //Button 3 - Select
 * ...
 *     // Here is the heart of the system: the navigator. 
 *     // The navigator takes in a reference to the root and a reference to an lcd
 *     Navigator navigator(&rootMenu, &lcd);
 *     
 *     // attach the methods for buttons Up, Down, Select to the navigator
 *     T1.attach_asserted( &navigator, &Navigator::moveUp);
 *     T2.attach_asserted( &navigator, &Navigator::moveDown);
 *     T3.attach_asserted( &navigator, &Navigator::select);
 *     // do whatever you need to do in your main-loop.
 *        while( 1 ) {
 *          led4 = !led4;
 *          wait( 1 );
 *        }
 * @endcode
*/
class Navigator
{

public:
    Navigator(Menu *, TextLCD_Base *);
    Menu *activeMenu; // the current menu - can change when Menue-item is selected on selection with child menu

    TextLCD_Base *lcd;

    /** Move up one line in menu.
    *   call this method when user moves up one line. 
    *   can be triggered by RPG or Button (PinDetect) or otherwise.
    */
    void moveUp();

    /** Move down one line in menu.
    *   call this method when user moves down one line.
    *   can be triggered by RPG or Button (PinDetect) or otherwise.
    */
    void moveDown();

    /** User presses Select Button.
    *   call this method when user wants to select an item.
    *   can be triggered by RPG or Button (PinDetect) or otherwise.
    */
    void select();

    /** print the menu on LCD
    */
    void printMenu();

    /** print cursor on the beginning of line
    */
    void printCursor();
    
private:
    /** Show Yes/No Dialog and wait fo Selection
    */
    void show_yes_no(bool yesorno);
    /** Show a long Text and wait for Select. Scroll Up/Down in Text
    */
    void show_longtext(void);

    int _display_rows; // number of rows the LCD can display
    int _display_cols; // number of lines of LCD
    int _cursorPos;    // what selection the cursor points to
    int _cursorLine;   // what line of the lcd the cursor is on. 1 = first line, 2 = second line
    int _start_line;   // display a long text starting from this line
    bool _wait_for_select;  // only accept Select Button to go Back
    bool _wait_for_yesno;   // up/don change selection ; Select accepts
    
};

#endif