/*
 * mbed LCDWindow library
* Copyright (c) 2010 Hendrik Lipka
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*/

#include "terminal.h"
#include "string.h"

Terminal::Terminal(Window* window):Window()
{
    _window=window;
    _columns=window->getColumns();
    _rows=window->getRows();
    _lineBuffer=new char*[_rows];
    _enabled = true;
    clear();
}

char* Terminal::createLine()
{
    char* text=new char[_columns+1];
    memset(text,32,_columns);
    text[_columns]=0;
    return text;
}

void Terminal::character(int column, int row, int c)
{
    if (column>_columns || row > _rows) {
        return;
    }
    _lineBuffer[row][column]=c;
    if (_enabled) {
        _window->character(column,row,c);
    }

}

void Terminal::writeText(const unsigned int column, const unsigned int row, const char text[])
{
    if (_enabled) {
        _window->writeText(column,row,text);
    }
    int min=column+strlen(text);
    if (min>_columns)
        min=_columns;
    for (int i=column; i<min; i++) {
        _lineBuffer[row][i]=text[i-column]; // copy text into proper line
    }
}

void Terminal::addText(const char text[])
{
    delete [] _lineBuffer[0];
    for (int i=0; i<_rows-1; i++) {
        _lineBuffer[i]=_lineBuffer[i+1];
    }
    _lineBuffer[_rows-1]=createLine();
    memset(_lineBuffer[_rows-1],32,_columns);
    int min=strlen(text);
    if (min>_columns)
        min=_columns;
    for (int i=0; i<min; i++) {
        _lineBuffer[_rows-1][i]=text[i]; // copy text into proper line
    }
    if (_enabled) {
        _window->clear();
        for (int i=0; i<_rows; i++) {
            _window->writeText(0,i,_lineBuffer[i]);
        }
    }
    // text longer than line?
    if (strlen(text) > _columns) {
        // add the remaining text to the next Line
        this->addText(text + _columns);
    }
}

void Terminal::clear()
{
    for (int i=0; i<_rows; i++) {
        _lineBuffer[i]=createLine();
    }
    if (_enabled) {
        _window->clear();
    }
}

void Terminal::disableDisplay()
{
    if (_enabled) {
        _enabled = false;
        _window->clear();
    }
}

void Terminal::enableDisplay()
{
    if (! _enabled) {
        _enabled = true;
        _window->clear();
        //Display the buffer-content
        for (int i=0; i<_rows; i++) {
            _window->writeText(0,i,_lineBuffer[i]);
        }
    }
}