/*
 * This VL53L3 Expansion board test application performs range measurements
 * using the onboard embedded sensor in interrupt mode.
 * Measured ranges are output on the Serial Port, running at 115200 baud.
 *
 * The Reset button can be used to restart the program.
 *
 * *** Note :
 * Default Mbed build system settings disable printf floating-point support.
 * Online builds seem unable to configure this.
 * Offline builds can enable printf floating-point support.
 * https://github.com/ARMmbed/mbed-os/blob/master/platform/source/minimal-printf/README.md
 * .\mbed-os\platform\mbed_lib.json
 *
 */

#include <stdio.h>
#include <time.h>

#include "mbed.h"

#include "vl53L3_I2c.h"
#include "vl53lx_platform_user_data.h"
#include "53l3a2_ranging_sensor.h"
#include "VL53L3A2_RangingClass.h"
#include "pinmap.h"

#define BSP_ERROR_NONE                         0


#if (MBED_VERSION  > 60300)
UnbufferedSerial  pc(USBTX, USBRX);
extern "C" void wait_ms(int ms);
#else
Serial pc(SERIAL_TX, SERIAL_RX);
#endif


#include "53l3a2_ranging_sensor.h"

VL53L3A2_SENSOR *sensor; //class for sensor commands

// interrupt pins for the three sensors.
// These are the default Pins but can be
// changed by adding solder blobs to the x-nucleo
PinName InterruptPins[] = {D8,A2,D2}; 



/* Private typedef -----------------------------------------------------------*/

/* Private define ------------------------------------------------------------*/
#define SENSOR_LEFT_NUMBER 0
#define SENSOR_CENTRE_NUMBER 1
#define SENSOR_RIGHT_NUMBER 2

/* Private variables ---------------------------------------------------------*/
static int32_t status = 0;


static const char *TofDevStr[] = {
    [VL53L3A2_DEV_LEFT] = "LEFT",
    [VL53L3A2_DEV_CENTER] = "CENTER",
    [VL53L3A2_DEV_RIGHT] = "RIGHT"
};

/* Private function prototypes -----------------------------------------------*/
static void MX_53L3A2_MultiSensorRanging_Init(void);
static void MX_53L3A2_MultiSensorRanging_Process(void);
static void print_result(uint8_t ToF_sensor,RANGING_SENSOR_Result_t *Result);

void MX_TOF_Process(void);

//flag to signal an interrupt has occured
volatile uint8_t EventDetected = 0;

int ToF_sensor = SENSOR_CENTRE_NUMBER;  // select the sensor to use


/* ISR callback function of the active sensor */
/* all it does set a flag which causes the main loop to get data. */
/* Interrupts activated and defined in CUSTOM_RANGING_ConfigProfile() */
void sensor_irq(void)
{
    EventDetected = 1;
}


// initialise the sensor
static void MX_53L3A2_MultiSensorRanging_Init(void)
{
    uint16_t i2c_addr;
    uint32_t id;

    // shut down all sensors
    sensor->VL53L3A2_RANGING_SetPowerMode(SENSOR_CENTRE_NUMBER, RANGING_SENSOR_POWERMODE_OFF);
    sensor->VL53L3A2_RANGING_SetPowerMode(SENSOR_LEFT_NUMBER, RANGING_SENSOR_POWERMODE_OFF);
    sensor->VL53L3A2_RANGING_SetPowerMode(SENSOR_RIGHT_NUMBER, RANGING_SENSOR_POWERMODE_OFF);
    wait_ms(100);

    printf("53L3A2 Single Sensor Ranging interrupt demo application\n");

    /* Power on the device, initialize it and change it's address.
     * Once the address is updated, the communication with the devices is checked by
     * reading its ID.
     */

    printf("VL53L3A2_RANGING_SENSOR_Init\n");
    sensor->VL53L3A2_RANGING_Init(ToF_sensor);
    if (status) {
        printf("VL53L3A2_RANGING_SENSOR_Init failed for sensor %d \n",ToF_sensor);
    }
    printf("VL53L3A2_RANGING_SENSOR_SetAddress\n");

    wait_ms(100);
    i2c_addr = (0x52 + (ToF_sensor + 1) * 2);	/* 0x54, 0x56, 0x58 */
    sensor->VL53L3A2_RANGING_SetAddress(ToF_sensor, i2c_addr);

    wait_ms(100);
}


// start ranging and enter an infinite loop to collect measurements
// The collection of data is triggered by a flag which is set after an interrupt occurs
static void MX_53L3A2_MultiSensorRanging_Process(void)
{

// config profile details
    RANGING_SENSOR_Result_t Result;
    RANGING_SENSOR_ProfileConfig_t Profile;
    printf("MX_53L3A2_MultiSensorRanging_Process\n");


    Profile.RangingProfile = RS_MULTI_TARGET_LONG_RANGE;
    Profile.TimingBudget = 30; /* 16 ms < TimingBudget < 500 ms */
    Profile.Frequency = 0; /* not necessary in simple ranging */
    Profile.EnableAmbient = 0; /* Enable: 1, Disable: 0 */
    Profile.EnableSignal = 0; /* Enable: 1, Disable: 0 */
    Profile.pin_gpio1 = InterruptPins[ToF_sensor]; // interrupt pin
    Profile.Interrupt_Func = sensor_irq;   // function that handles interrupts
    Profile.EnableInterrupt = 1;  // enables interupts

    printf("MX_53L3A2_MultiSensorRanging_Process configure sensors\n");
    sensor->VL53L3A2_RANGING_ConfigProfile(ToF_sensor, &Profile);
    if (status != BSP_ERROR_NONE) {
        printf("VL53L3A2_RANGING_SENSOR_ConfigProfile  failed sensor %d status %d\n",ToF_sensor,(int)status);
    }
    wait_ms(100);


    printf("VL53L3A2_RANGING_Start %d\n",ToF_sensor);
    status = sensor->VL53L3A2_RANGING_Start(ToF_sensor, RS_MODE_BLOCKING_CONTINUOUS);
    if (status != BSP_ERROR_NONE) {
        printf("VL53L3A2_RANGING_SENSOR_Start failed sensor %d status %d\n",ToF_sensor,(int)status);
    }


    EventDetected = 1;  // clear any existing interrupts
    
    // repeatedly read data and start next measurement
    while (1) {
        if ( EventDetected == 1 ) { // irq detected
            EventDetected = 0;
            status = sensor->VL53L3A2_RANGING_GetDistance(ToF_sensor, &Result); // gets the data and clears irq and starts next measurement
                      	  
            if ((status == BSP_ERROR_NONE) && 
                (Result.NumberOfZones != 0)) 
            {
	                print_result(ToF_sensor,&Result);
            }
        }

    }
}


// Note in VL53L3CX there is only one Zone so NumberOfZones is not used and is always assumed to be 1
static void print_result(uint8_t ToF_sensor,RANGING_SENSOR_Result_t *Result)
{
    uint8_t zone = 0; // in VL53L3CX there is only one zone
    uint8_t target = 0;

    for (target = 0; target < Result->ZoneResult[zone].NumberOfTargets; target++) {
        if (Result->ZoneResult[zone].Status[target] == VL53LX_RANGESTATUS_RANGE_VALID )
        {
	        printf("\n%s\t - ", TofDevStr[ToF_sensor]);
	        printf(" |---> ");
	        printf("Status = %d, Target %d, Distance = %5d mm",
	               Result->ZoneResult[zone].Status[target],
	               target,
	               Result->ZoneResult[zone].Distance[target]);
        }

    }
}



/*=================================== Main ==================================
=============================================================================*/
int main()
{

    pc.baud(115200);  // baud rate is important as printf statements take a lot of time

    sensor = new VL53L3A2_SENSOR();

    printf(">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>\r\n");
    printf("VL53L3CX_Shield_1Sensors_Interrupt\r\n");

    MX_53L3A2_MultiSensorRanging_Init(); // initialise the sensor

    while (1) {
        MX_53L3A2_MultiSensorRanging_Process(); // start ranging and collect data
    }

}

// needed for later version of mbed as wait_ms removed
#if (MBED_VERSION  > 60300)
extern "C" void wait_ms(int ms)
{
    thread_sleep_for(ms);
}
#endif

