/*
 * This VL53L3 Expansion board test application performs range measurements
 * using the onboard embedded sensor, and satellite boards, in interrupt mode.
 * Measured ranges are output on the Serial Port, running at 115200 baud.
 * Only supports one sensor at a time. Only tested on centre sensor.
 *
 * The Reset button can be used to restart the program.
 *
 * *** Note :
 * Default Mbed build system settings disable printf floating-point support.
 * Online builds seem unable to configure this.
 * Offline builds can enable printf floating-point support.
 * https://github.com/ARMmbed/mbed-os/blob/master/platform/source/minimal-printf/README.md
 * .\mbed-os\platform\mbed_lib.json
 *
 */

#include <stdio.h>
#include <time.h>

#include "mbed.h"

#include "vl53L3_I2c.h"
#include "vl53lx_platform_user_data.h"
#include "custom_ranging_sensor.h"
#include "Custom_RangingClass.h"
#include "pinmap.h"


#if (MBED_VERSION  > 60300)
UnbufferedSerial  pc(USBTX, USBRX);
extern "C" void wait_ms(int ms);
#else
Serial pc(SERIAL_TX, SERIAL_RX);
#endif

/* Private define ------------------------------------------------------------*/
#define SENSOR_LEFT_NUMBER 0
#define SENSOR_CENTRE_NUMBER 1
#define SENSOR_RIGHT_NUMBER 2




PinName InterruptPins[] ={D8,A2,D2}; // interrupt pins for the three sensors. 
                                     // these will depend on how the sensor is wired


/* Private variables ---------------------------------------------------------*/
static int32_t status = 0;
CUSTOM_SENSOR *sensor; //class for sensor commands


/* Private function prototypes -----------------------------------------------*/
static void MX_53L3A2_MultiSensorRanging_Init(void);
static void MX_53L3A2_MultiSensorRanging_Process(void);
static void print_result(RANGING_SENSOR_Result_t *Result);


//flag to signal an interrupt has occured
volatile uint8_t EventDetected = 0;

int ToF_sensor = CUSTOM_VL53L3CX;  // select the sensor to use. In custom_ranging_sensor.h


/* ISR callback function of the active sensor */
/* all it does set a flag which causes the main loop to get data. */
/* Interrupts activated and defined in CUSTOM_RANGING_ConfigProfile() */
void sensor_irq(void)
{
    EventDetected = 1;
}


// initialise the sensor
static void MX_53L3A2_MultiSensorRanging_Init(void)
{   
    status = sensor->CUSTOM_RANGING_Init(ToF_sensor);
    if (status)
    {
        printf("CUSTOM_RANGING_Init failed for sensor %d status %d\n",ToF_sensor,status);
    }
            
    wait_ms(100);
}


// start ranging and enter an infinite loop to collect measurements
// The collection of data is triggered by a flag which is set after an interrupt occurs
static void MX_53L3A2_MultiSensorRanging_Process(void)
{

// config profile details
    RANGING_SENSOR_Result_t Result;
    RANGING_SENSOR_ProfileConfig_t Profile;
    uint16_t zone = 0;
    uint16_t target = 0;
    printf("MX_53L3A2_MultiSensorRanging_Process\n");
    
    
    Profile.RangingProfile = RS_MULTI_TARGET_LONG_RANGE;
    Profile.TimingBudget = 30; /* 16 ms < TimingBudget < 500 ms */
    Profile.Frequency = 0; /* not necessary in simple ranging */
    Profile.EnableAmbient = 0; /* Enable: 1, Disable: 0 */
    Profile.EnableSignal = 0; /* Enable: 1, Disable: 0 */
    Profile.pin_gpio1 = InterruptPins[ToF_sensor]; // interrupt pin
    Profile.Interrupt_Func = sensor_irq;   // function that handles interrupts
    Profile.EnableInterrupt = 1;  // enables interupts

    printf("CUSTOM_RANGING_ConfigProfile configure sensors\n");
    sensor->CUSTOM_RANGING_ConfigProfile(ToF_sensor, &Profile);
    if (status != BSP_ERROR_NONE)
    {
      printf("CUSTOM_RANGING_ConfigProfile  failed sensor %d status %d\n",ToF_sensor,status);
    }
    wait_ms(100);
    

    printf("CUSTOM_RANGING_Start %d\n",ToF_sensor);
    status = sensor->CUSTOM_RANGING_Start(ToF_sensor, RS_MODE_BLOCKING_CONTINUOUS);
    if (status != BSP_ERROR_NONE)
    {
        printf("CUSTOM_RANGING_Start failed sensor %d status %d\n",ToF_sensor,status);
    }
 
    EventDetected = 1;  // clear any existing interrupts
    // repeatedly read data and start next measurement
    while (1)
    {
        if ( EventDetected == 1 ) // irq detected
        {
            EventDetected = 0;
            status = sensor->CUSTOM_RANGING_GetDistance(ToF_sensor, &Result);
            if ((status == BSP_ERROR_NONE) && 
                ( Result.ZoneResult[zone].NumberOfTargets != 0))  // is there data
            {
                for ( target = 0 ; target < Result.ZoneResult[zone].NumberOfTargets; target++)
                { 
                    if (Result.ZoneResult[zone].Status[target] == VL53LX_RANGESTATUS_RANGE_VALID )
                    {
                        printf("\n |---> ");
                        printf("Status = %d, Target %d, Distance = %5d mm",
                               Result.ZoneResult[zone].Status[target],
                               target,
                               Result.ZoneResult[zone].Distance[target]);
                    }
                }
            }
        }
      
    }
}



static void print_result(RANGING_SENSOR_Result_t *Result)
{
    uint8_t j = 0;
    uint8_t zone = 0;
    
    printf(" |---> ");
    printf("Status = %d, Distance = %5d mm \n",
            Result->ZoneResult[zone].Status[j],
            Result->ZoneResult[zone].Distance[j]);
         
}
  


/*=================================== Main ==================================
=============================================================================*/
int main()
{
    
    pc.baud(115200);  // baud rate is important as printf statements take a lot of time
    
    sensor = new CUSTOM_SENSOR();
    
    printf(">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>\r\n");
    printf("VL53L3CX_NoShield_1Sensors_interrupt\r\n");   

    MX_53L3A2_MultiSensorRanging_Init();
        
    while (1)
    {
         MX_53L3A2_MultiSensorRanging_Process();
    }

}

#if (MBED_VERSION  > 60300)
extern "C" void wait_ms(int ms)
{
    thread_sleep_for(ms);
}
#endif
  