/*
 * This VL53L3 Expansion board test application performs range measurements
 * using the onboard embedded sensor, and satellite boards, in interrupt mode.
 * Measured ranges are output on the Serial Port, running at 115200 baud.
 *
 * The Reset button can be used to restart the program.
 *
 * *** Note :
 * Default Mbed build system settings disable printf floating-point support.
 * Online builds seem unable to configure this.
 * Offline builds can enable printf floating-point support.
 * https://github.com/ARMmbed/mbed-os/blob/master/platform/source/minimal-printf/README.md
 * .\mbed-os\platform\mbed_lib.json
 *
 */

#include <stdio.h>
#include <time.h>

#include "mbed.h"

#include "vl53L3_I2c.h"
#include "vl53lx_platform_user_data.h"
#include "custom_ranging_sensor.h"
#include "Custom_RangingClass.h"


#define BSP_ERROR_NONE                         0


#if (MBED_VERSION  > 60300)
UnbufferedSerial  pc(USBTX, USBRX);
extern "C" void wait_ms(int ms);
#else
Serial pc(SERIAL_TX, SERIAL_RX);
#endif


CUSTOM_SENSOR *sensor; //class for sensor commands


/* Private define ------------------------------------------------------------*/
#define POLLING_PERIOD              (250U)

/* Private variables ---------------------------------------------------------*/
static int32_t status = 0;
int ToF_sensor = CUSTOM_VL53L3CX;  // select the sensor to use. In custom_ranging_sensor.h


/* Private function prototypes -----------------------------------------------*/
void MX_TOF_Process(void);



void MX_TOF_Init(void)
{

    uint16_t i2c_addr;
    uint32_t id;

    status = sensor->CUSTOM_RANGING_Init(ToF_sensor);
    if (status) {
        printf("CUSTOM_RANGING_Init failed for sensor %d status %d\n",ToF_sensor,status);
    }

}



void MX_TOF_Process(void)
{

    RANGING_SENSOR_Result_t Result;
    RANGING_SENSOR_ProfileConfig_t Profile;
    uint16_t zone = 0;
    uint16_t target = 0;

    printf("MX_53L3A2_MultiSensorRanging_Process\n");
    Profile.RangingProfile = RS_MULTI_TARGET_LONG_RANGE;
    Profile.TimingBudget = 30; /* 16 ms < TimingBudget < 500 ms */
    Profile.Frequency = 0; /* not necessary in simple ranging */
    Profile.EnableAmbient = 0; /* Enable: 1, Disable: 0 */
    Profile.EnableSignal = 0; /* Enable: 1, Disable: 0 */
    Profile.EnableInterrupt = 0;  // disables interupts
    printf("MX_53L3A2_MultiSensorRanging_Process start sensors\n");

    wait_ms(100);
//	sensor->VL53L3A2_RANGING_ConfigProfile(ToF_sensor, &Profile);
    sensor->CUSTOM_RANGING_ConfigProfile(ToF_sensor, &Profile);
    if (status != BSP_ERROR_NONE) {
        printf("CUSTOM_RANGING_ConfigProfile  failed sensor %d status %d\n",ToF_sensor,status);
    }

    wait_ms(100);
    status = sensor->CUSTOM_RANGING_Start(ToF_sensor, RS_MODE_BLOCKING_CONTINUOUS);
    if (status != BSP_ERROR_NONE) {
        printf("CUSTOM_RANGING_Start failed sensor %d status %d\n",ToF_sensor,status);
    }

    printf("MX_53L3A2_MultiSensorRanging_Process read sensors\n");

    // repeatedly read data and start next measurement
    while (1) {
        status = sensor->CUSTOM_RANGING_GetDistance(ToF_sensor, &Result);
      // print results  	  
        if ((status == BSP_ERROR_NONE) && 
            ( Result.ZoneResult[zone].NumberOfTargets != 0))  // is there data
        {
        	for ( target = 0 ; target < Result.ZoneResult[zone].NumberOfTargets; target++)
        	{ 
	        	if (Result.ZoneResult[zone].Status[target] == VL53LX_RANGESTATUS_RANGE_VALID )
	        	{
		            printf("\n |---> ");
		            printf("Status = %d, Target %d, Distance = %5d mm",
		                   Result.ZoneResult[zone].Status[target],
		                   target,
		                   Result.ZoneResult[zone].Distance[target]);
	            }
	        }
        }

        wait_ms(POLLING_PERIOD);
    }
}


/*=================================== Main ==================================
=============================================================================*/
int main()
{

    pc.baud(115200);  // baud rate is important as printf statements take a lot of time

    printf("53L3A2 One Sensor Ranging demo application\n");

    sensor = new CUSTOM_SENSOR();

    MX_TOF_Init();

    while (1) {
        MX_TOF_Process();
    }

}

// needed for later versions of mbed
#if (MBED_VERSION  > 60300)
extern "C" void wait_ms(int ms)
{
    thread_sleep_for(ms);
}
#endif

