/*
 * @file DS1721.h
 * @author Cameron Haegle
 *
 * @section LICENSE
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * Library for the DS1721, 2-Wire Digital Thermometer and Thermostat, from Maxim Semiconductor
 *  (www.maiximintegrated.com). 
 *
 * @section LIMITATIONS
 * 
 * This library was not written with for use with the mbed-RTOS, as the I2C read/write calls are 
 *  not thread safe.
 *
 * ChangeLog
 * 07/30/2013 
 *   - 
 */

#ifndef DS1721_H
#define DS1721_H

#include "mbed.h"

// default DS1721 I2C address
#define DS1721_ADDR             0x48

// DS1721 commands
#define CMD_START_CONVT         0x51
#define CMD_STOP_CONVT          0x22
#define CMD_READ_TEMP           0xAA
#define CMD_ACCESS_CFG          0xAC
#define CMD_ACCESS_TH           0xA1
#define CMD_ACCESS_TL           0xA2

// thermometer config register values
#define CONV_FOREVER            0x00
#define CONV_ONE_SHOT           0x01
#define POLARITY_ACTIVE_LOW     0x00
#define POLARITY_ACTIVE_HIGH    0x02
#define RES_9_BIT               0x00    // 0000
#define RES_10_BIT              0x04    // 0100
#define RES_11_BIT              0x08    // 1000
#define RES_12_BIT              0x0C    // 1100

// DS1721 operating modes (HEAT = active low, Cool = active high)
#define MODE_HEAT               POLARITY_ACTIVE_LOW
#define MODE_COOL               POLARITY_ACTIVE_HIGH


/**
 * !Library for the DS1721 temperature sensor.
 * The DS1721 is an I2C digital temperature sensor, with a range of -55C to +125C and a 0.125C resolution.
 * 
 * @code
 * #include "mbed.h"
 * #include "DS1721.h"
 * 
 * #define DS1721_I2C_ADDRESS (0x48<<1)
 * 
 * int main(void) {
 * I2C m_i2c(PTC9, PTC8);
 * DS1721 thermo(m_i2c, DS1721_I2C_ADDRESS);
 * 
 * // initialize the temperature sensor
 * thermo.startConversion();
 * thermo.setLowSp(25);
 * thermo.setHighSp(27);
 * thermo.setPolarity(POLARITY_ACTIVE_HIGH);
 * 
 *     while (true) {       
 *         printf("Temp: %d\r\n", thermo.getTemp());
 *     }
 * } 
 * @endcode
 **/
class DS1721
{
public:
  /** 
   * DS1721 constructor
   *
   * @param &i2c pointer to i2c object
   * @param addr addr of the I2C peripheral default = (DS1721_ADDR<<1)
   **/
  DS1721(I2C &i2c, int addr = (DS1721_ADDR<<1));
  
  /** 
   * DS1721 constructor
   * 
   * @param &i2c pointer to I2C object
   * @param resolution readout resolution of the theremometer (9, 10, 11, 12 bit)
   * @param polarity the state on which the thermostat output is enabled
   * @param mode the temperature conversion mode (single shot or continuous)
   * @param addr addr of the I2C peripheral default = (DS1721_ADDR<<1)
   **/
  DS1721(I2C &i2c,int resolution, int polarity, int mode, int addr = (DS1721_ADDR<<1));
  
  /*!
   * DS1721 destructor
   */ 
  ~DS1721();
  
  /**
   * Reads the temperature from the DS1721 and converts it to a useable value.
   *
   * @returns the current temperature, as a float
   **/
  float getTemp(void);

  /**
   * Sets the temperature polarity (POL) bit of the sensor. This bit determines 
   * upon which set point the theremostat output is active.
   *
   * @param int the new polarity value
   * @returns the result of the function (0 - Failure, 1 - Success) 
   **/  
  int setPolarity(int polarity);

  /**
   * Gets the temperature polarity (POL) bit of the sensor. This bit determines 
   * upon which set point the theremostat output (Tout) is active.
   *
   * @returns the value of the POL bit (0 - active low, 1 - active high)
   **/    
  int getPolarity(void);

  /**
   * Initiates a temperature conversion. If the DS1721 is configured for single shot mode 
   * this function must be called prior to reading the temperature (re: getTemp())
   *
   * @returns the result of the function (0 - Failure, 1 - Success) 
   **/    
  int startConversion(void);

  /**
   * Stops a temperature conversion. 
   *
   * @returns the result of the function (0 - Failure, 1 - Success) 
   **/   
  int stopConversion(void);

  /**
   * Sets the configuration register of the DS1721. 
   *
   * @param int the new configuration value (resolution | polarity | mode)
   * @returns the result of the function (0 - Failure, 1 - Success) 
   **/    
  int setConfig(int config);
  
  /**
   * Retrieves the configuration register of the DS1721.
   *
   * @returns 
   **/    
  int getConfig(int* config);

  /**
   * Retrieves the configured low temperature set point value.
   *
   * @returns the current low temperature set point value (degrees C)
   **/    
  float getLowSp(void);

  /**
   * Sets the low temperature set point value.
   *
   * @param int the new low set point temperature (degrees C)
   * @returns the result of the function (0 - Failure, 1 - Success) 
   **/    
  int setLowSp(float newSp);

  /**
   * Retrieves the configured high temperature set point value.
   *
   * @returns the current high temperature set point value (degrees C)
   **/    
  float getHighSp(void);

  /**
   * Sets the high temperature set point value.
   *
   * @param int the new high temperature set point value (degrees C)
   * @returns the result of the function (0 - Failure, 1 - Success) 
   **/    
  int setHighSp(float newSp);
  
  /*
   * Helper function to convert degrees Celcius to Fahrenheit
   * 
   * @param temperature in degrees Celcius
   * @return temperature in degrees Fahrenheit
   */
  float temp_CtoF(float tempC);
  
  /*
   * Helper function to convert degrees Fahrenheit to Celcius
   * 
   * @param temperature in degrees Fahrenheit
   * @return temperature in degrees Celcius
   */
  float temp_FtoC(float tempF);

  
private:
  I2C &m_i2c;
  int m_addr;
  int _resolution;
  int _polarity;
  int _mode;
  int _tl;
  int _num_read;
  int _th;
  
  int writeData(char* data, int length);
  int readData(char* data, int length);
};

#endif

