
#define GND_TMP_SENSOR A0
#define HUM_MAX 24500.0
#define HUM_MIN 20600.0

#include "mbed.h"
#include "ADA326.h"
#include "DS1820.h"
//#include "glibr.h"
#include "LinkedList.h"
#include "DHT.h"
#include "FSM.h"
#include "HMI.h"
#include "Sensor.h"
#include "Data.h"

// Sleep
bool modeSleep;
// Time events
Ticker ticker;
Timer timer;
Ticker chrono;
unsigned int sec = 0;

// Communication
bool usingI2C;
Serial pc(USBTX, USBRX);
I2C i2c(I2C_SDA, I2C_SCL);
#define _SIGFOX_
#ifdef _SIGFOX_
Serial sg(D1, D0); //Sigfox
#endif


/**** Declaration for ****/
// LED & Interrupt
DigitalOut led(D13);

// Human Machine Inteface
InterruptIn button(D10);
FSM fsm_button(FSM_Button::NB_BUTTON_STATE);
Ada326 lcd;
FSM fsm_display(FSM_Display::NB_DISPLAY_STATE);

// Sensors - see Sensor.h to activate sensors with macros.
#ifdef _GND_SENSOR_
DS1820 gndTemperatureSensor(GND_TMP_SENSOR); 
AnalogIn gndHumSensor(A2);
#endif

#ifdef _COLOR_SENSOR_
glibr rgbSensor; // liaison de type I2C avec le module de mesure de lumière
#endif

#ifdef _DHT_SENSOR_
DHT dht(A1, DHT22);
#endif
////////////////////////////////////////////

// Interrupt events
void onRisingEdge(void);
void count_3600s(void);

// Intialization
void initialize_device(); // Initialize sensors
void initialize_fsm(); // Add finite states / transitions.

/* 
Repeat the function pointed by the function pointer "fnc" every "interval" seconds.
During the wait, the microcontroler will be on standby mode for low consumption of energy.
*/
void repeat(Callback<void()> event, bool &wait, unsigned int interval);

void measure(); // Activate sensors to measure surroundings and read data.
void transmission_sigfox(); // send data read from sensors using sigfox


int main()
{
    bool wait_sensor = false, wait_sigfox = false;
    // Waiting time in seconds until ...
    // next measurement, using sensors:
    int SENSOR_SLEEP_TIME = 600;
    int SIGFOX_SLEEP_TIME = 600;

    if(SENSOR_SLEEP_TIME >= 3599 || SIGFOX_SLEEP_TIME >= 3599) {
        pc.printf("Waiting too long to activate sensors/sigfox");
        return -1;
    }
        
    // Initialize.
    usingI2C = false;
    modeSleep = true; 
    initialize_fsm();
    initialize_device();
    
    // Start timer (measurement in second)
    chrono.attach(&count_3600s,1);
    // Attach a function to be called at button interrupt event.
    button.rise(&onRisingEdge);
    
    while(1) {
        // Call "measure" to activate sensors every SENSOR_SLEEP_TIME
        repeat(&measure, wait_sensor, SENSOR_SLEEP_TIME);
        // Call "transmission_sigfox" to send data every SENSOR_SLEEP_TIME
        repeat(&transmission_sigfox, wait_sigfox, SIGFOX_SLEEP_TIME);
        
        /* 
        Execute the task of the current state of FSM.
        Transition will be made automatically if the transition is triggered by
        conditions imposed by each state.
        
        cf. the FSM explained at our website.
        
        FSM is implemented to debounce button; turn on/off and change the display.
        */
        
        // fsm_button <= implemented for debounce.
        fsm_button.execute();
        fsm_display.execute();
    }
    return 0;
}

void repeat(Callback<void()> event, bool &wait, unsigned int interval)
{
    if(sec % interval == 0 && !wait) { // Time to call event. 
        //Prevent repeated calling within a second by setting the flag, "wait".
        event();
        wait = true;
    }
    else if(sec % interval) { // Time to be on standby
        if(modeSleep) sleep();
        wait = false;
    }
}
void measure() 
{
    // Signal that sensors are activated.
    // It hints that button and display can't be used during this time.
    if(display_on) {
        lcd.clear(false);
        lcd.print("En cours...");  
        lcd.display();
    }
    // Deactivate button / screen.
    usingI2C = true;  
    
    // Start measurement of ...
    
    // 1. Ground sensors.
    #ifdef _GND_SENSOR_
    gndTemperature = getGndTemperature();
    gndHumidity = 100 - (((float)(getGndHumidity() - HUM_MIN)) / 30148.0) * 100;
    #endif
    
    // 2. Light sensors        
    #ifdef _COLOR_SENSOR_
    getLightValues(&ambientLight, &redLight, &greenLight, &blueLight); // récupère les données du capteur de lumière 
    #endif
    
    // 3. Air sensors
    #ifdef _DHT_SENSOR_
    dht.readData();
    airHumidity = dht.ReadHumidity();
    airTemperature = dht.ReadTemperature(CELCIUS);
    #endif
    
    // Done activate button / screen
    usingI2C = false;
    if(display_on) {
        renew_screen = true;
        next_screen = false; 
    }
}
void transmission_sigfox()
{
    //pc.printf("Start sigfox.\r\n");     
#ifdef _SIGFOX_
    // Encrypt and compress the measurement data in 3trame3.
    struct SensorData data = {gndHumidity, (double)gndTemperature, airHumidity, (double)airTemperature, ambientLight, redLight, greenLight, blueLight};
    char text[150];
    int trame[3];
    create_frame(trame, data); 
    sprintf(text, "AT$SF=%08x%08x%08x\n\r", trame[2] , trame[1] , trame[0]);
    
    sg.printf(text);        
#endif
    //pc.printf(text);        
    //pc.printf("Finish sigfox.\r\n");     
}


void initialize_device()
{
    //lcd.initialize();
    lcd.setFontSize(0.21f);
    //lcd.clear(true);
    
//Initaliise the sensors.
#ifdef _GND_SENSOR_
    init_gndTmp();
#endif
#ifdef _COLOR_SENSOR_
    init_rgbSensor();
#endif
}
// Button interrupt event :
void onRisingEdge(void) 
{
    
    if (usingI2C) return;
    
    modeSleep = false;
    if(!RE_detected)
        RE_detected = true;
}
void count_3600s(void) 
{
    // Reset counter at every 3600 (It never reaches 3600)
    sec = (sec + 1) % 3600;
} 
void initialize_fsm()
{
    fsm_button.setInitialState(FSM_Button::State_Idle);
    fsm_button.addState(FSM_Button::State_Idle, FSM_Button::button_idle);
    fsm_button.addState(FSM_Button::State_Debounce, FSM_Button::button_debounce);
    fsm_button.addState(FSM_Button::State_Pressed, FSM_Button::button_pressed);
    fsm_button.addState(FSM_Button::State_Released, FSM_Button::button_released);
    fsm_button.addState(FSM_Button::State_Wait, FSM_Button::button_wait);
    
    fsm_display.setInitialState(FSM_Display::State_Idle);
    fsm_display.addState(FSM_Display::State_Idle, FSM_Display::display_idle);
    fsm_display.addState(FSM_Display::State_Activate, FSM_Display::display_activate);
    fsm_display.addState(FSM_Display::State_Write, FSM_Display::display_write);
    fsm_display.addState(FSM_Display::State_Standby, FSM_Display::display_standby);
}