/*
 * mbed ST7565R Display Driver
 * Copyright (c) 
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef ST7565R_H
#define ST7565R_H

#include "Text.h"
#include "mbed.h"

typedef struct
{
    char screen1[512];
    char screen2[512];
} screen_memory_t;

typedef struct
{
    short x;
    short y;
} loc_t;
/** ST7565R Display Driver; uses some code taken from the TextLCD driver for displaying text.
  *
  * Example:
  * @code
  * #include "ST7565R.h"
  * #include "mbed.h"
  *
  * ST7565R lcd(p12,p29,p7,p5, p30 , false);
  * 
  * int main() { 
  *   lcd.printf("Hello world!");
  * }
  * @endcode
  */
class ST7565R : public Stream
{
public:
    enum {
        BITBLT_SRCCOPY,
        BITBLT_SRCAND,
        BITBLT_SRCOR
    };
   /** Create a ST7565R display object on the pins specified.
    *
    * @param cs  The chip select pin for the display.
    * @param a0  The data/read select line for the display; specified as A0 in the datasheet.
    * @param scl The clock pin for SPI.
    * @param rst The reset pin on the display.
    * @param BackBuffer The display I have has a memory area thats can be used for backbuffering. This turns this option on. When using this option you must call swapBuffers() in order to display your changes to the screen. This is to help prevent flickering on the display. If not using a backbuffer all writes using bitblt or setpixel are immediate and visible to the screen.
    */
    ST7565R(PinName cs,PinName a0, PinName scl, PinName si, PinName rst, bool BackBuffer);
    
   /** Function to move the text cursor to where specified; 
    * DISCLAIMER: All text routines are based on code from TextLCD, who's link is in the cookbook.
    *
    * @param col  The column number. With the font in Text.h this number is 21 and can be retrieved from columns()
    * @param row  The chip select pin for the display.
    */
    void moveto(int col, int row);
    
   /** Function to send a command byte to the display.
    *
    * @param value The command byte to send over spi.
    */
    void command(int value);
    
   /** Function to send display data to the display.
    *
    * @param value The data byte to send over spi. Memory is arranged in pages. A data write will cover 8 pixels vertically (the page) and only one column. A data write will increment the column by 1 but will never increment the page.
    */
    void data(int value);
    
   /** Function that uses the reset line and then reinitializes the display as well as clearing the current buffer.
    */
    void reset();
    
   /** Function to set a specific pixel to black.
    * @param x Measured from the top left of the display; If the display is in backbuffering mode the pixel change is completely within the offscreen buffer and won't display until swapBuffer() is called.
    * @param y Measured from the top left of the display; If the display is in backbuffering mode the pixel change is completely within the offscreen buffer and won't display until swapBuffer() is called.
    */
    void setpixel(int x, int y);

   /** Function to set a specific pixel to white.
    * @param x Measured from the top left of the display; If the display is in backbuffering mode the pixel change is completely within the offscreen buffer and won't display until swapBuffer() is called.
    * @param y Measured from the top left of the display; If the display is in backbuffering mode the pixel change is completely within the offscreen buffer and won't display until swapBuffer() is called.
    */    
    void clearpixel(int x, int y);
   /** Function to get the buffer being used inside the object to do bit writes, required because of the lack of reading screen data back over spi for the ST7565R. The buffer returned is the one that you can draw to without immediately updating the screen.
    */        
    char * getDrawingScreen();
   /** Function to get the buffer being used inside the object to do bit writes, required because of the lack of reading screen data back over spi for the ST7565R. The buffer returned is the one that has the current screen data on it. Without backbuffering mode this returns the current screen.
    */         
    char * getCurrentScreen();
   /** Function to clear the current drawing buffer. This clears the current screen when not using a backbuffer.
    */         
    void clearBuffer();
   /** Function to commit all the data to the drawing buffer, then calls the display command to change buffers so as to not promote a flicker in the display. Not needed if not using a backbuffer.
    */         
    void swapBuffers();
   /** Function to blit an image to the screen. Used internally for copying the font to the screen, but also is powerful for displaying any monochrome image.
    * @param dstx The x coordinate destination on the screen you want to blit to.
    * @param dsty The y coordinate destination on the screen you want to blit to.
    * @param blit_width The width of the image copy you want to perform.
    * @param blit_height The height of the image copy you want to perform. 
    * @param img The pointer to the data of the image.
    * @param img_width The width of the image. Required for calculating the address to start the next line in the img buffer. 
    * @param img_height The height of the image. 
    * @param srcx The x coordinate where to start copying image data from within the source image.
    * @param srcy The y coordinate where to start copying image data from within the source image
    * @param format What kind of copy to perform. BITBLT_SRCCOPY is a standard image copy.
    */    
    void bitblt(int dstx, int dsty, 
                int blit_width, int blit_height, 
                char* img, 
                int img_width, int img_height,
                int srcx, int srcy, 
                int format);

   /** Function to place a character on the screen. 
    * DISCLAIMER: All text routines are based on code from TextLCD, who's link is in the cookbook.
    *
    * @param column  The column number. With the font in Text.h this number is 21 and can be retrieved from columns()
    * @param row  The row number. With the font in Text.h this number is 4 and can be retrieved from rows()
    */
    void character(int column, int row, char c);
   /** Function to return the number of columns; for now all it does is returns 21. Reserved for possible later code manipulation with multiple fonts.
    * DISCLAIMER: All text routines are based on code from TextLCD, who's link is in the cookbook.
    */
    int columns();
   /** Function to return the number of rows; for now all it does is returns 4. Reserved for possible later code manipulation with multiple fonts.
    * DISCLAIMER: All text routines are based on code from TextLCD, who's link is in the cookbook.
    */
    int rows();
protected:
   /** Overwritten function from the Stream object.
    * DISCLAIMER: All text routines are based on code from TextLCD, who's link is in the cookbook.
    */
    virtual int _putc(int value);
   /** Overwritten function from the Stream object.
    * DISCLAIMER: All text routines are based on code from TextLCD, who's link is in the cookbook.
    */
    virtual int _getc();
private:
    int _column;
    int _row;

    SPI        _spi;
    DigitalOut _cs;
    DigitalOut _a0;
    DigitalOut _rst;

    bool scr;

    char * current_buffer;
    loc_t pos;
    screen_memory_t screen;  
    bool UsingBackBuffer;  
};

#endif