/* mbed 4D uOLED Library
 * Originally designed for use with uOLED-96-G1 (SGC)
 * serially controlled .96" screen.
 *
 * This is a modified library originally obtained from
 * Erik van Wijk's library code at:
 * http://mbed.org/users/evwijk/libraries/microOLED/li4nzn
 *
 * This library (uOLED) by Aaron Goselin.
 * 2010
 *
 * adaptation for uOLED-128-G1 by Christian Dupaty 02-2021
 *
 * If you use and/or modify this library please keep credit lines intact.
 */

#ifndef _MBED_UOLED_
#define _MBED_UOLED_

#include "mbed.h"
#include "GSGCdef.h"

#define     OLED_FONT5X7                    0x0
#define     OLED_FONT8X8                    0x1
#define     OLED_FONT8X12                   0x2

#define     OLED_DISPLAYCONTROL_DISPLAY     0x01
#define     OLED_DISPLAYCONTROL_CONTRAST    0x02
#define     OLED_DISPLAYCONTROL_POWER       0x03


/** uOLED control class using Serial
 * 
 * The serially controlled uOLEDs by 4D Systems are controlled
 * with only 3 pins:
 * - serialTX
 * - serialRX
 * - reset
 *
 * While the device includes many serial functions, it is faster to do things from a uSD card.
 * Consider learning the 4DSL scripting language.  You can then write your functions in 4DSL 
 * storing them on the uSD to later be triggered serially.
 *
 * Examples use SGC as the uOLED instance name.  SGC just happens to be the type of device that I have.
 * Of course, it doesn't matter what you call your instance(s).
 *
 * Examples use both decimal and hexadecimal numbers.  It does not matter which you use for most functions.
 *
 * Please post a comment on the library page if you spot an error.  I will try to fix it quickly.
 *
 * Example:
 * @code
 * // Draw text on the screen.
 * #include "mbed.h"
 * #include "uOLED.h"
 * 
 * uOLED SGC(p9, p10, p11);
 *
 * int main()
 * {    
 *     SGC.drawText(0, 0, 0, FF, FF, FF, "This is text");
 * }
 * @endcode
 */
class uOLED {
public:

    /** Create an instance of the uOLED class.
    * 
    * @param serialTX - mbed TX pin to be used
    * @param serialRX - mbed RX pin to be used
    * @param reset - mbed pin to control reset of the uOLED
    *
    * Example:
    * @code
    * // (serialTX, serialRX, reset)
    * uOLED SGC(p9, p10, p11);
    * @endcode
    */
    uOLED(PinName serialTX, PinName serialRX, PinName reset);
    
    /** Convert RGB value into the type of value that the uOLED wants.
    * 
    * @param red Red value.
    * @param green Green value.
    * @param blue Blue value.
    *
    * Example:
    * @code
    * // (red, green, blue)
    * SGC.getRGB(120, 255, 50);
    * @endcode
    */
    short getRGB(char red, char green, char blue);

    /** Add user defined bitmap character into internal memory.
    * 
    * @param character Character index to add to memory.  Range is 0-31 (0x00 to 0x1F).  32 8x8 characters.
    * @param data[8] 8 data bytes that make up the composition of the bitmap character.  8x8 composition is 1 byte wide (8 bits) by 8 bytes deep.
    *
    * Example:
    * @code
    * // Array containing data for 8x8 "O" character.
    * char data[8] = {0x18, 0x24, 0x42, 0x81, 0x81, 0x42, 0x24, 0x18};
    * // (characterIndex, characterData)
    * SGC.addBitmappedCharacter(0x01, data);
    * @endcode
    */
    bool addBitmappedCharacter(char character, char data[8]);
    
    /** Copy and paste a specified block of the screen.
    * 
    * @param sourceX Top left horizontal start position of screen area to be copied.
    * @param sourceY Top left vertical start position of screen area to be copied.
    * @param destinationX Top left horizontal start position of where copied area is to be pasted.
    * @param destinationY Top left vertical start position of where copied area is to be pasted.
    * @param width Width of screen area to be copied.
    * @param height Height of screen area to be copied.
    *
    * Example:
    * @code
    * // (sourceX, sourceY, destinationX, destinationY, width, height)
    * SGC.blockCopyPaste(0, 0, 25, 5, 10, 10);
    * @endcode
    */
    bool blockCopyPaste(char sourceX, char sourceY, char destinationX, char destinationY, char width, char height);
    
    /** Save a specified block of the screen to the uSD.
    * 
    * @param x Top left horizontal start position of screen area to be copied.
    * @param y Top left vertical start position of screen area to be copied.
    * @param width Width of screen area to be copied.
    * @param height Height of screen area to be copied.
    * @param sectorHi High sector address.
    * @param sectorMid Mid sector address.
    * @param sectorLow Low sector address.
    *
    * Example:
    * @code
    * // Save a block of the screen from (0x, 0y) to (20x, 20y) and save it on the uSD
    * // at (0x00, 0x10, 0x66)
    * // (x, y, width, height, sectorHi, sectorMid, sectorLow)
    * SGC.screenCopy(0, 0, 20, 20, 0x00, 0x10, 0x66);
    * @endcode
    */
    bool screenCopy(char x, char y, char width, char height, char sectorHi, char sectorMid, char sectorLow);
    
    
    /** Display control.
    * 
    * Display ON/OFF, set contrast, and PowerUp/Shutdown.
    *
    * @param mode Sets specified mode.
    * @param value Value option for mode.
    *
    * - Mode: 01 Display ON/OFF
    *   - Value: 00 OFF
    *   - Value: 01 ON
    * - Mode: 02 Contrast Adjust
    *   - Value range: 0x00 to 0x0F
    * - Mode: 03 Display PowerUp/Shutdown (low power mode)
    *   - Value 00 Shutdown
    *   - Value 01 Powerup
    *
    * Example:
    * @code
    * // Turn display ON.
    * SGC.displayControl(0x01, 0x01)
    * // Set contrast to medium.
    * SGC.displayControl(0x02, 0x08);
    * // Shutdown display.
    * SGC.displayControl(0x03, 0x00);
    * @endcode
    */
    bool displayControl(char mode, char value);
    
    /** Draw previously defined user bitmap character at specified location (and colour).
    * 
    * @param character Character index of previously defined bitmap character.  Range is 0 to 31 (0x00 to 0x1F).  32 8x8 characters.
    * @param x Horizontal display position of character.
    * @param y Vertical display position of character.
    * @param red Amount of red.
    * @param green Amount of green.
    * @param blue Amount of blue.
    *
    * Example:
    * @code
    * // Display bitmapped character stored in index 01, at location (0x,0y), and colour it red.
    * // (characterIndex, x, y, red, green, blue)
    * SGC.displayUserBitmappedCharacter(0x01, 0x00, 0x00, 0xFF, 0x00, 0x00);
    * @endcode
    */
    bool displayUserBitmappedCharacter(char character, char x, char y, short red, short green, short blue);
    
    /** Draw a circle.
    * 
    * @param x X position of circle center.
    * @param y Y position of circle center.
    * @param radius Radius of the circle.
    * @param red Amount of red.
    * @param green Amount of green.
    * @param blue Amount of blue.
    *
    * Example:
    * @code
    * // Draw a circle centered at (63x, 63y) with a radius of 34 and colour it green.
    * // (x, y, radius, red, green, blue)
    * SGC.drawCircle(0x3F, 0x3F, 0x22, 0x00, 0xFF, 0x00);
    * @endcode
    */
    bool drawCircle(char x, char y, char radius, short red, short green, short blue);
    
    /** Draw ASCII character (text format)
    * 
    * @param character Inbuilt standard ASCII character.  Range 32 to 127 (0x20 to 0x7F).
    * @param column Horizontal position of character.
    * - range: 0-20 for 5x7 font.
    * - range: 0-15 for 8x8 and 8x12 fonts.
    * @param row Vertical position of character.
    * @param red Amount of red.
    * @param green Amount of green.
    * @param blue Amount of blue.
    *
    * Example:
    * @code
    * // Draw character 'A' at column 0, row 0.  Colour it white.
    * // (character, column, row, red, green, blue)
    * SGC.drawCharacter(0x41, 0x00, 0x00, 0xFF, 0xFF, 0xFF);
    * @endcode
    */
    bool drawCharacter(char character, char column, char row, short red, short green, short blue);
    
    /** Display a bitmap image on the screen at specified location and size.
    * 
    * @param x Image horizontal start position (top left).
    * @param y Image vertical start position (top left).
    * @param width Horizontal size of the image.
    * @param height Vertical size of the image.
    * @param colorMode Colour mode to use for the image.
    * - 0x08 = 256 colour mode (8bits/1byte per pixel)
    * - 0x10 = 65K colour mode (16bits/2bytes per pixel)
    * @param *pixels Image pixel data.
    * - Colour Mode 0x08 (256 colour mode):  Number of pixels = width x height
    * - Colour Mode 0x10 (65K colour mode):  Number of pixels = 2 x width x height
    */
    bool drawImage(char x, char y, char width, char height, char colorMode, char *pixels);
    
    /** Load an image from the uSD and display on the screen at specified location and size.
    * 
    * @param x Image horizontal start position (top left).
    * @param y Image vertical start position (top left).
    * @param width Horizontal size of the image.
    * @param height Vertical size of the image.
    * @param colorMode Colour mode to use for the image.
    * - 0x08 = 256 colour mode (8bits/1byte per pixel)
    * - 0x10 = 65K colour mode (16bits/2bytes per pixel)
    * @param sectorHi High sector address.
    * @param sectorMid Mid sector address.
    * @param sectorLow Low sector address.
    */
    bool drawImageSD_16bit(char x, char y, char width, char height, char colourMode, char sectorHi, char sectorMid, char sectorLow);
    
    
    /** Draw a line.
    * 
    * @param x1 Top left horizontal start position.
    * @param y1 Top left vertical start position.
    * @param x2 Bottom right horizontal start position.
    * @param y2 Bottom right vertical end position.
    * @param red Amount of red.
    * @param green Amount of green.
    * @param blue Amount of blue.
    *
    * Example:
    * @code
    * // Draw a line starting at (0x, 0y) and ending at (43x, 43y).  Colour it blue.
    * // (x1, y1, x2, y2, red, green, blue)
    * SGC.drawLine(0x00, 0x00, 0x2B, 0x2B, 0x00, 0x00, 0xFF);
    * @endcode
    */
    bool drawLine(char x1, char y1, char x2, char y2, short red, short green, short blue);
    
    /** Draw a polygon (user defined shape) to the screen.
    * 
    * @param vertices Number of vertices from 3 to 7.
    * @param *x Array of vertices' X coordinates.
    * @param *y Array of vertices' Y coordinates.
    * @param red Amount of red.
    * @param green Amount of green.
    * @param blue Amount of blue.
    *
    * Example:
    * @code
    * char x[5] = {0, 18, 26, 44, 54};
    * char y[5] = {10, 25, 33, 22, 36};
    * // Draw a white polygon with 5 vertices located at:
    * // (0x, 10y), (18x, 25y), (26x, 33y), (44x, 22y), (54x, 36y)
    * // (vertices, *x, *y, red, green, blue)
    * SGC.drawPolygon(5, x, y, 255, 255, 255);
    * @endcode
    */
    bool drawPolygon(char vertices, char *x, char *y, short red, short green, short blue);
    
    /** Draw a rectangle.
    * 
    * @param x Top left horizontal start position.
    * @param y Top left vertical start position.
    * @param width Bottom right horizontal end position.
    * @param height Bottom right vertical end position.
    * @param red Amount of red.
    * @param green Amount of green.
    * @param blue Amount of blue.
    *
    * Example:
    * @code
    * // Draw rectangle starting at (0x, 0y) and ending at (40x, 40y).  Colour red.
    * // (x, y, width, height, red, green, blue)
    * SGC.drawRectangle(0, 0, 40, 40, 255, 0, 0);
    * @endcode
    */
    bool drawRectangle(char x, char y, char width, char height, short red, short green, short blue);
    
    /** Draw text to the screen.
    * 
    * @param column X coordinate for text placement.
    * @param row Y coordinate for text placement.
    * @param font Which font to use (Uses 0, 1, or 2).  More fonts can be added.
    * @param color Font colour to use.
    * @param *text Character array (string) to be displayed.
    *
    * Example:
    * @code
    * // Draw string "This is text" at (0, 0) with font set 0, coloured white.
    * // (column, row, font, red, green, blue, "text")
    * SGC.drawText(0, 0, 0, 0xFF, 0xFF, 0xFF, "This is text");
    * @endcode
    */
    bool drawText(char column, char row, char font, short red, short green, short blue, char *text);
    
    /** Draw unformated text to the screen.
    * The manual describes this as "graphics format".
    *
    * @param x X coordinate for text placement.
    * @param y Y coordinate for text placement.
    * @param font Which font to use (Uses 0, 1, or 2).  More fonts can be added.
    * @param red Amount of red in text colour RGB value.
    * @param green Amount of green in text colour RGB value.
    * @param blue Amount of blue in text colour RGB value.
    * @param width Text width.
    * @param height Text height.
    * @param *text Character array (string) to be displayed.
    *
    * Example:
    * @code
    * // Draw unformatted text string "This is text" at (0, 0) with font set 0, coloured white with zero (1x) magnification.
    * SGC.drawTextUF(0, 0, 0, 255, 255, 255, 1, 1, "This is text");
    * @endcode
    */
    bool drawTextUF(char x, char y, char font, short red, short green, short blue, char width, char height, char *text);
    
    /** Draw a triangle.
    *
    * Vertices must be defined counter clockwise.
    *
    * @param x1 Vertice 1 X coordinate.
    * @param y1 Vertice 1 Y coordinate.
    * @param x2 Vertice 2 X coordinate.
    * @param y2 Vertice 2 Y coordinate.
    * @param x3 Vertice 3 X coordinate.
    * @param y3 Vertice 3 Y coordinate.
    * @param red Amount of red.
    * @param blue Amount of blue.
    * @param green Amount of green.
    *
    * Example:
    * @code
    * // Draw a red triangle with vertices:
    * // (0x, 0y), (0x, 40y), (40x, 0y)
    * // (x1, y1, x2, y2, x3, y3, red, green, blue)
    * SGC.drawTriangle(0, 0, 0, 40, 40, 0, 255, 0, 0);
    * @endcode
    */
    bool drawTriangle(char x1, char y1, char x2, char y2, char x3, char y3, short red, short green, short blue);
    
    /** Clear the screen.
    * 
    * Remove everything except the background colour.
    *
    * @param returns success or failure.
    *
    * Example:
    * @code
    * // Do you really need an example? :)
    * SGC.eraseScreen();
    * @endcode
    */
    bool eraseScreen();
    
    /** Initialize the screen.  This must be completed before any other communication with the device.
    *
    * Timing allows for at least 500ms delay for initialization.
    *
    * @param returns bool indicating success or failure of initialization.
    *
    * Example:
    * @code
    * // Must be done before anything else
    * SGC.init();
    * @endcode
    */
    bool init();
    
    /** Set pen size.
    * 
    * Sets if objects should be drawn solid or wire frame.
    * Does not apply to polygon function.
    *
    * @param size Sets solid or wire frame.
    * - 0x00 = Solid
    * - 0x01 = Wire frame.
    *
    * Example:
    * @code
    * // Draw objects solid
    * SGC.penSize(0);
    * // Draw objects wire frame
    * SGC.penSize(1);
    * @endcode
    */
    bool penSize(char size);
    
    /** Draw a coloured pixel at designated position.
    * 
    * @param x Horizontal position of pixel.
    * @param y Vertical position of pixel.
    * @param red Amount of red.
    * @param green Amount of green.
    * @param blue Amount of blue.
    *
    * Example:
    * @code
    * // Draw a blue pixel at (10x, 10y).
    * // (x, y, red, green, blue)
    * SGC.putPixel(10, 10, 0, 0, 255);
    * @endcode
    */
    bool putPixel(char x, char y, short red, short green, short blue);
    
    /** Read the colour of a specified pixel.
    * 
    * @param x X coordinate.
    * @param y Y coordinate.
    * @param returns 2 byte pixel colour in RGB format.
    * - MSB: R4R3R2R1R0G5G4G3
    * - LSB: G2G1G0B4B3B2B1B0
    *
    * Example:
    * @code
    * // Read pixel colour at location (20x, 20y).
    * short pixelColour = SGC.readPixel(20, 20);
    * @endcode
    */
    short readPixel(char x, char y);
    
    /** Replaces the background colour with a new colour.
    * 
    * Most functions call this internally.
    *
    * @param red Red value (0 to 255).
    * @param green Green value (0 to 255).
    * @param blue Blue value (0 to 255).
    *
    * Example:
    * @code
    * // Set background colour to red.
    * // (red, green, blue)
    * SGC.setBackgroundColour(255, 0, 0);
    * @endcode
    */
    bool setBackgroundColour(char red, char green, char blue);
    
    /** Set font (for future text).
    *
    * 3 default fonts are supplied.  Further font sets can be added.
    * 
    * @param font Font selection.  Either a default or otherwise.
    * - DEFAULT: 0x00 = 5x7 small size font set
    * - DEFAULT: 0x01 = 8x8 medium size font set
    * - DEFAULT: 0x02 = 8x12 large size font set
    *
    * Example:
    * @code
    * // Use default 5x7 font set
    * SGC.setFont(0);
    * // Use default 8x12 font set
    * SGC.setFont(2);
    * @endcode
    */    
    bool setFont(char font);
    
    /** Draw a text button to the screen.
    * 
    * @param state Button down (0x00) or button up (0x01).
    * @param x Top left horizontal start position.
    * @param y Top left vertical start position.
    * @param red Amount of red (for button not text).
    * @param green Amount of green (for button not text).
    * @param blue Amount of blue (for button not text).
    * @param font Set which font set to use for the button.  3 default font sets are supplied and more can be added.
    * - DEFAULT: 0x00 = 5x7 small size font set
    * - DEFAULT: 0x01 = 8x8 medium size font set
    * - DEFAULT: 0x02 = 8x12 large size font set
    * @param textRed Amount of red for text.
    * @param textGreen Amount of green for text.
    * @param textBlue Amount of blue for text.
    * @param textWidth Width of characters (text).  Affect total width of string and button.
    * @param textHeight Height of characters (text).  Affects total height of string and button.
    * @param *text Character array (string) to display in the button.
    *
    * Example:
    * @code
    * // Draw a text button in the unpressed state starting at (0x, 0y).
    * // Button colour is red, text colour is white.  Text magnification is zero (1x).
    * // (state, x, y, red, green, blue, font, textRed, textGreen, textBlue, textWidth, textHeight, "Text");
    * SGC.textButton(1, 0, 0, 255, 0, 0, 0, 255, 255, 255, 1, 1, "Button text");
    * @endcode
    */
    bool textButton(char state, char x, char y, short red, short green, short blue, char font, short textRed, short textGreen, short textBlue, char textWidth, char textHeight, char *text);
    
    /** Set text mode (transparent or opaque).
    *
    * Opaque text has a rectangle drawn behind it (blocking out what is under it) and transparent text does not.
    * 
    * @param mode Set text to transparent (0x00) or opaque (0x01).
    *
    * Example:
    * @code
    * // Set text mode to transparent.
    * SGC.textMode(0);
    * @endcode
    */
    bool textMode(char mode);
    
    /** Retrieve current version info of the device.
    * 
    * Response:
    * 
    * device_type Indicates device type.
    * - 0x00 = micro-OLED
    * - 0x01 = micro-LCD
    * - 0x02 = micro-VGA
    *
    * hardware_rev Indicates device hardware version.
    *
    * firmware_rev Indicates device firmware version.
    *
    * horizontal_res Indicates the horizontal resolution of the display.
    * - 0x22 = 220 pixels
    * - 0x28 = 128 pixels
    * - 0x32 = 320 pixels
    * - 0x60 = 160 pixels
    * - 0x64 = 64 pixels
    * - 0x76 = 176 pixels
    * - 0x96 = 96 pixels
    *
    * vertical_res Indicates the vertical resolution of the display.
    * - See horizontal_res (identical).
    *
    * @param onScreen Set output option.
    * - 0x00 = Output to serial port only.
    * - 0x01 = Output to serial port and screen.
    * @param *info Character array to store results.
    *
    * Example:
    * @code
    * // Request version info, pass in character array to store results.
    * char info[5];
    * SGC.versionInfo(1, info);
    * printf("\n\nVersion info:\n\n");
    * for(int i = 0; i < 5; i++)
    * {
    *    printf("0x%X\n", info[i]);
    * }
    * @endcode
    */
    bool versionInfo(bool onScreen, char info[5]);
    
    
protected:
    Serial      _oled;
    DigitalOut  _reset;
    
    void resetDisplay();
};

#endif