/***************************************************
  This is an example sketch for our optical Fingerprint sensor
  Designed specifically to work with the Adafruit BMP085 Breakout
  ----> http://www.adafruit.com/products/751
  These displays use TTL Serial to communicate, 2 pins are required to
  interface
  Adafruit invests time and resources providing this open source code,
  please support Adafruit and open-source hardware by purchasing
  products from Adafruit!
  Written by Limor Fried/Ladyada for Adafruit Industries.
  BSD license, all text above must be included in any redistribution
  
  STM32 adaptation by Christian Dupaty 03/2021
 ****************************************************/

#include <Fingerprint.h>
#include "mbed.h"

DigitalOut ledV(LED1); // for various information without terminal
Serial pc(USBTX, USBRX); // tx, rx
Fingerprint finger(PC_1,PC_0,0x0); // TX,TX,pass
InterruptIn fD(PB_0);               // IT from fingerprint detection (see datasheet WAKEUP)
DigitalIn btnBleu(USER_BUTTON);     // to start enroll

bool fingerON=false;

// interrupt funtion 
void fingerDetect(void)
{
    ledV=1;
    wait_ms(100); 
    ledV=0;
    fingerON=true;   
}


// original setup fonction on Arduino
void setup()
{
  pc.baud(9600);
  wait_ms(100);
  pc.printf("\nR503 Finger detect test\nSTM32 version with MBED compiler and library\n");

  // set the data rate for the sensor serial port
  finger.begin(57600);
  wait_ms(200);
  if (finger.verifyPassword()) {
    pc.printf("\nFound fingerprint sensor!\n");
  } else {
    pc.printf("\nDid not find fingerprint sensor -> STOP !!!!\n");
    while (1)
    {
         ledV=1;
         wait_ms(100); 
         ledV=0;
         wait_ms(100); 
    }
  }

  pc.printf("Reading sensor parameters\n");
  finger.getParameters();
  pc.printf("Status: 0x%X\n",finger.status_reg);
  pc.printf("Sys ID: 0x%X\n",finger.system_id);
  pc.printf("Capacity: %d\n",finger.capacity);
  pc.printf("Security level: %d\n",finger.security_level);
  pc.printf("Device address: 0x%X\n",finger.device_addr);
  pc.printf("Packet len: %d\n",finger.packet_len);
  pc.printf("Baud rate: %d\n",finger.baud_rate);

  finger.getTemplateCount();

  if (finger.templateCount == 0) {
    pc.printf("Sensor doesn't contain any fingerprint data. Please run the 'enroll' example.\n");
  }
  else {
    pc.printf("Waiting for valid finger...\n");
    pc.printf("Sensor contains : %d templates\n",finger.templateCount);
  }
  fD.fall(&fingerDetect);  
}

// --------------------------------------
uint8_t getFingerprintID() {
  uint8_t p = finger.getImage();
  switch (p) {
    case FINGERPRINT_OK:
      pc.printf("Image taken\n");
      break;
    case FINGERPRINT_NOFINGER:
      pc.printf("No finger detected\n");
      return p;
    case FINGERPRINT_PACKETRECIEVEERR:
      pc.printf("Communication error\n");
      return p;
    case FINGERPRINT_IMAGEFAIL:
      pc.printf("Imaging error\n");
      return p;
    default:
      pc.printf("Unknown error\n");
      return p;
  }

  // OK success!

  p = finger.image2Tz();
  switch (p) {
    case FINGERPRINT_OK:
      pc.printf("Image converted\n");
      break;
    case FINGERPRINT_IMAGEMESS:
      pc.printf("Image too messy\n");
      return p;
    case FINGERPRINT_PACKETRECIEVEERR:
      pc.printf("Communication error\n");
      return p;
    case FINGERPRINT_FEATUREFAIL:
      pc.printf("Could not find fingerprint feature\n");
      return p;
    case FINGERPRINT_INVALIDIMAGE:
      pc.printf("Could not find fingerprint features\n");
      return p;
    default:
      pc.printf("Unknown error\n");
      return p;
  }

  // OK converted!
  p = finger.fingerSearch();
  if (p == FINGERPRINT_OK) {
    pc.printf("Found a print match!\n");
  } else if (p == FINGERPRINT_PACKETRECIEVEERR) {
    pc.printf("Communication error\n");
    return p;
  } else if (p == FINGERPRINT_NOTFOUND) {
    pc.printf("Did not find a match\n");
    return p;
  } else {
    pc.printf("Unknown error\n");
    return p;
  }

  // found a match!
  pc.printf("Found ID #%d\n",finger.fingerID);
  pc.printf(" with confidence of %d\n",finger.confidence);

  return finger.fingerID;
}

// returns -1 if failed, otherwise returns ID #
int getFingerprintIDez() {
  uint8_t p = finger.getImage();
  if (p != FINGERPRINT_OK)  return -1;

  p = finger.image2Tz();
  if (p != FINGERPRINT_OK)  return -1;

  p = finger.fingerFastSearch();
  if (p != FINGERPRINT_OK)  return -1;

  // found a match!
  pc.printf("Found ID #%d\n",finger.fingerID);
  pc.printf(" with confidence of %d\n",finger.confidence);
  return finger.fingerID;
}

void demoLED(void)
{
// control (3 on)(4off), speed (0-255) , color (1 red, 2 blue, 3 purple), cycles (0 infinit,- 255)
// red
    finger.LEDcontrol(3,128,1,10);
    wait_ms(500);  
// blue
    finger.LEDcontrol(3,128,2,10);  
    wait_ms(500); 
// cyan
    finger.LEDcontrol(3,128,3,10);
    wait_ms(500);   
// flash 5
    finger.LEDcontrol(2,10,1,5);
    wait_ms(1000);  
// slow light on
    finger.LEDcontrol(5,50,2,5);
    wait_ms(1000);  
// flash slow
    finger.LEDcontrol(1,50,2,5);
    wait_ms(1000); 
}

int main(void)                    // run over and over again
{
unsigned char c=1;
  setup();
  demoLED();
  finger.LEDcontrol(3,128,1,10);
  pc.printf("\nPret ! \n");
  while(1)
  {

      if (fingerON)
      {
          pc.printf("Doigt detecte ! \n");     
          fingerON=false;
          getFingerprintID();
          pc.printf("\nPret ! \n");
      }
      if (!btnBleu)
      {
        pc.printf("\nEnregistrement empreinte \n");
        c=finger.createModel();
        if (c==FINGERPRINT_OK) pc.printf("Empreinte enregistree  \n");
        if (c==FINGERPRINT_PACKETRECIEVEERR) pc.printf("Erreur de communication \n");
        if (c==FINGERPRINT_ENROLLMISMATCH) pc.printf("Erreur de scan \n");
        pc.printf("\nPret ! \n");
      }
      wait_ms(500);            //don't need to run this at full speed.
  }
 // return 0;
}

