/*
    Copyright (c) 2010 Andy Kirkham
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
*/

#ifndef MODOSD7456_H
#define MODOSD7456_H

#include "mbed.h"
#include "MAX7456.h"

/** defgroup OSD7456_API */

#define OSD_NUM_LINES   13
#define OSD_LINE_LEN    30

typedef struct _osd_line {
    char line[OSD_LINE_LEN];
    char attrib[OSD_LINE_LEN];
    bool updated;
} OSD_LINE;

/** OSD7456 module
 *
 * The OSD7456 is a wrapper around the MAX7456 that abstracts the hardware
 * into a simple to use screen writer system. It provides buffered output.
 * To ensure a "flicker free" display, the buffers are written to the MAX7456
 * chip at the start of the vertical sync period.
 *
 * @see http://mbed.org/cookbook/
 * @see example2.cpp
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "OSD7456.h"
 * 
 * DigitalOut led1(LED1);
 * 
 * OSD7456 *osd;
 * 
 * int main() {
 *     
 *     osd = new OSD7456(p5, p6, p7, p8, p20, p15);
 *     
 *     osd->print(1, "Hello World");
 *     osd->print2("This blinks", MAX7456::Blink);
 *     osd->print3("Background", MAX7456::LocalBG);    
 *     osd->print(5, 4, "Positioned");
 *     osd->print(5, 5, "Positioned", MAX7456::LocalBG);
 *     osd->print(3, 7, "Test");
 *     osd->print(12, 7, "Test", MAX7456::LocalBG | MAX7456::Blink);
 *     
 *     while(1) {
 *         led1 = 1;
 *         wait(0.5);        
 *         led1 = 0;
 *         wait(0.5);    
 *     }
 * }
 * @endcode
 */
 
class OSD7456
{
public:

    //! OSD7456 constructor.
    /**
     * The OSD7456 constructor.
     *
     * @ingroup OSD7456_API
     * @param miso PinName p5 or p11
     * @param mosi PinName p6 or p12
     * @param sclk PinName p7 pr p13
     * @param name Optional const char * SSP object name
     * @param cs   PinName CS signal
     * @param rst  PinName RESET signal
     * @param vsync PinName Vertical sync signal
     */
    OSD7456(PinName mosi, PinName miso, PinName sclk, const char *name, PinName cs, PinName rst, PinName vsync) {
        max7456 = new MAX7456(mosi, miso, sclk, name, cs, rst, vsync); 
        init();       
    }
    
    //! OSD7456 constructor.
    /**
     * The OSD7456 constructor.
     *
     * @ingroup OSD7456_API
     * @param miso PinName p5 or p11
     * @param mosi PinName p6 or p12
     * @param sclk PinName p7 pr p13
     * @param cs   PinName CS signal
     * @param rst  PinName RESET signal
     * @param vsync PinName Vertical sync signal
     */
    OSD7456(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName rst, PinName vsync) {
        max7456 = new MAX7456(mosi, miso, sclk, NULL, cs, rst, vsync);        
        init();
    }
    
    //! print()
    /**
     * Print ASCII text at line.
     *
     * @ingroup OSD7456_API
     * @param line The line number to print at.
     * @param s A pointer to the null terminated string to print.
     */
    int print(int line, char *s);
    
    //! print()
    /**
     * Print ASCII text on line y at position x
     *
     * @ingroup OSD7456_API
     * @param x The line position to print at.
     * @param y The line to print on.
     * @param s A pointer to the null terminated string to print.
     */
    int print(int x, int y, char *s);
    
    //! print()
    /**
     * Print ASCII text on line y at position x with attribute.
     *
     * @ingroup OSD7456_API
     * @param x The line position to print at.
     * @param y The line to print on.
     * @param s A pointer to the null terminated string to print.
     * @param a An attribute byte to apply to the string.
     */
    int print(int line, char *s, char a);
    
    //! print()
    /**
     * Print ASCII text on line y at position x
     *
     * @ingroup OSD7456_API
     * @param x The line position to print at.
     * @param y The line to print on.
     * @param s A pointer to the null terminated string to print.
     * @param a An attribute byte to apply to the string.
     */
    int print(int x, int y, char *s, char attrib);    
    
    //! clear()
    /**
     * Clear the line number supplied.
     *
     * @ingroup OSD7456_API
     * @param line The line number to clear.
     */
    void clear(int line);
    
    /** @ingroup OSD7456_API */
    /** @{ */
    __INLINE int print0(char *s) { return print(0, s); }
    __INLINE int print1(char *s) { return print(1, s); }
    __INLINE int print2(char *s) { return print(2, s); }
    __INLINE int print3(char *s) { return print(3, s); }
    __INLINE int print4(char *s) { return print(4, s); }
    __INLINE int print5(char *s) { return print(5, s); }
    __INLINE int print6(char *s) { return print(6, s); }
    __INLINE int print7(char *s) { return print(7, s); }
    __INLINE int print8(char *s) { return print(8, s); }
    __INLINE int print9(char *s) { return print(9, s); }
    __INLINE int print10(char *s) { return print(10, s); }
    __INLINE int print11(char *s) { return print(11, s); }
    __INLINE int print12(char *s) { return print(12, s); }
    __INLINE int print13(char *s) { return print(13, s); }
    
    __INLINE int print0(char *s, char a) { return print(0, s, a); }
    __INLINE int print1(char *s, char a) { return print(1, s, a); }
    __INLINE int print2(char *s, char a) { return print(2, s, a); }
    __INLINE int print3(char *s, char a) { return print(3, s, a); }
    __INLINE int print4(char *s, char a) { return print(4, s, a); }
    __INLINE int print5(char *s, char a) { return print(5, s, a); }
    __INLINE int print6(char *s, char a) { return print(6, s, a); }
    __INLINE int print7(char *s, char a) { return print(7, s, a); }
    __INLINE int print8(char *s, char a) { return print(8, s, a); }
    __INLINE int print9(char *s, char a) { return print(9, s, a); }
    __INLINE int print10(char *s, char a) { return print(10, s, a); }
    __INLINE int print11(char *s, char a) { return print(11, s, a); }
    __INLINE int print12(char *s, char a) { return print(12, s, a); }
    __INLINE int print13(char *s, char a) { return print(13, s, a); }
    
    /** @} */
    
    // The function registered with teh MAX7456 vertical sync interrupt handler.
    void vsync(void);
    
    /**
     * Attach a user callback object/method to call when the vsync signal activates. 
     *
     *
     * @ingroup OSD7456_API
     * @param tptr pointer to the object to call the member function on
     * @param mptr pointer to the member function to be called
     */
    template<typename T>
    void attach_vsync(T* tptr, void (T::*mptr)(void)) { cb_vsync.attach(tptr, mptr); }
    
    /**
     * Attach a user callback function pointer to call when the vsync signal activates. 
     *
     * @ingroup OSD7456_API
     * @param fptr Callback function pointer
     */
    void attach_vsync(void (*fptr)(void)) { cb_vsync.attach(fptr); } 
    
    //! A callback object for the 1PPS user API.
    FunctionPointer cb_vsync;
    
    MAX7456 *max7456;
    
protected:
    
    OSD_LINE lines[OSD_NUM_LINES];
    
    void init(void); 
    
    bool oddEven;   
};

#endif
