/**
 * Copyright (c) 2017, Arm Limited and affiliates.
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Adaptation for Discovery IOT STmicro B-L072Z-LRWAN1, TTN and  CAYENNE (mydevices.com)
 * Christian Dupaty Lycee Fourcade 13120 Gardanne France
 * 11-2018
 * Peripherals on B-L072Z-LRWAN1
    LED1        = PB_5, // Green
    LED2        = PA_5, // Red
    LED3        = PB_6, // Blue
    LED4        = PB_7, // Red
    USER_BUTTON = PB_2,
  */
#include <stdio.h>

#include "lorawan/LoRaWANInterface.h"
#include "lorawan/system/lorawan_data_structures.h"
#include "events/EventQueue.h"
#include "CayenneLPP.h"

// Application helpers
#include "DummySensor.h"
#include "trace_helper.h"
#include "lora_radio_helper.h"

using namespace events;

// Max payload size can be LORAMAC_PHY_MAXPAYLOAD.
// This example only communicates with much shorter messages (<30 bytes).
// If longer messages are used, these buffers must be changed accordingly.
uint8_t tx_buffer[30];
uint8_t rx_buffer[30];

/*
 * Sets up an application dependent transmission timer in ms. Used only when Duty Cycling is off for testing
 */
#define TX_TIMER                        10000

/**
 * Maximum number of events for the event queue.
 * 10 is the safe number for the stack events, however, if application
 * also uses the queue for whatever purposes, this number should be increased.
 */
#define MAX_NUMBER_OF_EVENTS            10

/**
 * Maximum number of retries for CONFIRMED messages before giving up
 */
#define CONFIRMED_MSG_RETRY_COUNTER     3

/**
 * Dummy pin for dummy sensor
 */
#define PC_9                            0

/**
 * Dummy sensor class object
 */
DS1820  ds1820(PC_9);

// added by C.Dupaty

CayenneLPP cayenne(100);   // data to cayenne format
AnalogIn lm35(A0);  // for test we have connect a 
                    //LM35 temperature captor on A0 Arduino analog port
AnalogIn ana(A1);
DigitalIn bp(USER_BUTTON);
DigitalOut ledTest(LED1);

/**
* This event queue is the global event queue for both the
* application and stack. To conserve memory, the stack is designed to run
* in the same thread as the application and the application is responsible for
* providing an event queue to the stack that will be used for ISR deferment as
* well as application information event queuing.
*/
static EventQueue ev_queue(MAX_NUMBER_OF_EVENTS * EVENTS_EVENT_SIZE);

/**
 * Event handler.
 *
 * This will be passed to the LoRaWAN stack to queue events for the
 * application which in turn drive the application.
 */
static void lora_event_handler(lorawan_event_t event);

/**
 * Constructing Mbed LoRaWANInterface and passing it down the radio object.
 */
static LoRaWANInterface lorawan(radio);

/**
 * Application specific callbacks
 */
static lorawan_app_callbacks_t callbacks;

/**
 * Entry point for application
 */
int main (void)
{
    // setup tracing
    setup_trace();

    // stores the status of a call to LoRaWAN protocol
    lorawan_status_t retcode;

    // Initialize LoRaWAN stack
    if (lorawan.initialize(&ev_queue) != LORAWAN_STATUS_OK) {
        printf("LoRa initialization failed! \r\n");
        return -1;
    }
    printf("\r\nTEST LORAWAN Lycee Fourcade \r\n");
    printf("--------------------------- \r\n");
    printf("\r\nMbed LoRaWANStack initialized \r\n");

    // prepare application callbacks
    callbacks.events = mbed::callback(lora_event_handler);
    lorawan.add_app_callbacks(&callbacks);

    // Set number of retries in case of CONFIRMED messages
    if (lorawan.set_confirmed_msg_retries(CONFIRMED_MSG_RETRY_COUNTER)
                                          != LORAWAN_STATUS_OK) {
        printf("set_confirmed_msg_retries failed! \r\n\r\n");
        return -1;
    }

    printf("CONFIRMED message retries : %d \r\n",
           CONFIRMED_MSG_RETRY_COUNTER);

    // Enable adaptive data rate
    if (lorawan.enable_adaptive_datarate() != LORAWAN_STATUS_OK) {
        printf("enable_adaptive_datarate failed! \r\n");
        return -1;
    }

    printf("Adaptive data  rate (ADR) - Enabled \r\n");

    retcode = lorawan.connect();

    if (retcode == LORAWAN_STATUS_OK ||
        retcode == LORAWAN_STATUS_CONNECT_IN_PROGRESS) {
    } else {
        printf("Connection error, code = %d \r\n", retcode);
        return -1;
    }

    printf("Connection - In Progress ...\r\n");
    printf("it may take a while\r\n");

    // make your event queue dispatching events forever
    ev_queue.dispatch_forever();

    return 0;
}

/**
 * Sends a message to the Network Server
 */
static void send_message()
{
    uint16_t packet_len;
    int16_t retcode;
    float sensor_value; // dummy
    float lum=0.0;      // dummy
    float hum;         // dummy
    float temp;         // dummy
    float an;           // AN1 on Arduino connectors
    int btBleu;         // The bleu button on Nucleo board
    
    float tempLM35;
    static int cpt=0;

    printf("\n\rEmission : %d\r\n",++cpt);
        printf("-------------\r\n");
    // Dummy sensors are used for test only
    if (ds1820.begin()) {
        ds1820.startConversion();
        sensor_value = ds1820.read();
        printf("Dummy Sensor Value = %3.1f \r\n", sensor_value);
        ds1820.startConversion();
    } else {
        printf("No sensor found \r\n");
        return;
    }
    lum+=111.0;
    if (lum>9999.0) lum=0.0;
    hum+=2.5;
    if (temp>100.0) temp=1.1;
    an=ana.read()*100.0;
    btBleu=bp.read();
    
    // LM35 must be connected on A0 analog port
    tempLM35=lm35.read()*330.0;
    printf("Capteur LM35 : %3.2f C\r\n", tempLM35);
    

    // mise en forme format CAYENNE LPP
    // doc ici https://mydevices.com/cayenne/docs/lora/#lora-cayenne-low-power-payload
    // doc logiciel : CayenneLPP.h
    cayenne.reset();
    cayenne.addLuminosity(1, lum);
    cayenne.addPresence(2,1);
    cayenne.addTemperature(3, temp);
    cayenne.addRelativeHumidity(4, hum);
    cayenne.addDigitalInput(5, btBleu);
    cayenne.addAnalogInput(6, an);
    cayenne.addDigitalOutput(7,0);
    cayenne.addTemperature(8, tempLM35);   
    cayenne.addTemperature(9, sensor_value);

    cayenne.copy(tx_buffer);
    
    packet_len=cayenne.getSize();
    printf("Emission du packet : \n\r");
    for (int i=0;i<packet_len;i++) printf("%02X ",tx_buffer[i]);
    printf("\n\r");
  //  packet_len = sprintf((char*) tx_buffer, "Dummy Sensor Value is %3.1f",          sensor_value);

    retcode = lorawan.send(MBED_CONF_LORA_APP_PORT, tx_buffer, packet_len,
                           MSG_CONFIRMED_FLAG);

    if (retcode < 0) {
        retcode == LORAWAN_STATUS_WOULD_BLOCK ? printf("send - WOULD BLOCK\r\n")
                : printf("send() - Error code %d \r\n", retcode);

        if (retcode == LORAWAN_STATUS_WOULD_BLOCK) {
            //retry in 3 seconds
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                ev_queue.call_in(3000, send_message);
            }
        }
        return;
    }

    printf("%d bytes scheduled for transmission \r\n", retcode);
    memset(tx_buffer, 0, sizeof(tx_buffer));
}

/**
 * Receive a message from the Network Server
 */
static void receive_message()
{
    int16_t retcode;
    uint8_t port; // var to store port number provided by the stack
    int flags; // var to store flags provided by the stack
    retcode = lorawan.receive( rx_buffer,sizeof(rx_buffer), port, flags);
    printf("\x1B[1m");  // yellow text
    if (retcode < 0) {
        printf("receive() - Error code %d \r\n", retcode);
        return;
    }
    printf(" Reception on port : %d \n",port);
    printf(" Flags are : %d \n",flags);
    printf(" Data: ");

    for (uint8_t i = 0; i < retcode; i++) {
        printf("%02X ", rx_buffer[i]);
    }

    printf("\n\r Data Length: %d\r\n", retcode);
    printf("\x1B[0m");  // white text
    printf("End reception\n\r");

    if (rx_buffer[2]==0x64) ledTest=1;
    if (rx_buffer[2]==0x00) ledTest=0;

    memset(rx_buffer, 0, sizeof(rx_buffer));
}

/**
 * Event handler
 */
static void lora_event_handler(lorawan_event_t event)
{
    switch (event) {
        case CONNECTED:
            printf("Connection - Successful \r\n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            } else {
                ev_queue.call_every(TX_TIMER, send_message);
            }

            break;
        case DISCONNECTED:
            ev_queue.break_dispatch();
            printf("Disconnected Successfully \r\n");
            break;
        case TX_DONE:
            printf("Message Sent to Network Server \r\n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            }
            break;
        case TX_TIMEOUT:
        case TX_ERROR:
        case TX_CRYPTO_ERROR:
        case TX_SCHEDULING_ERROR:
            printf("Transmission Error - EventCode = %d \r\n", event);
            // try again
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            }
            break;
        case RX_DONE:
            printf("Received message from Network Server \r\n");
            receive_message();
            break;
        case RX_TIMEOUT:
        case RX_ERROR:
            printf("Error in reception - Code = %d \r\n", event);
            break;
        case JOIN_FAILURE:
            printf("OTAA Failed - Check Keys \r\n");
            break;
        case UPLINK_REQUIRED:
            printf("Uplink required by NS \r\n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            }
            break;
        default:
            MBED_ASSERT("Unknown Event");
    }
}

// EOF
