/*
Christian Dupaty 10/2021
IS31FL3731 with CHARLEPLEX LED MATRIX adaptation from Adafruit project to ARM MBEB, tested on NUCLEO L073RZ
for original project on Arduino see https://learn.adafruit.com/animated-flame-pendant/overview

The program reads the data from an image and places it alternately
on pages 0 and 1 of IS31FL3731

Data structure in data.h
first byte x1 PF and y1 pf
second byte x2 PF and y2 pf
loop from x1 to x2
     loop from y1 to y2
         copy from the third byte into the img buffer (144 bytes)

the I2C write method sends the write address then a series of data, the number of data is then specified
ex: Wire.write (I2C_ADDR, cmd, s); I2C_ADDR address on 8bits, cmd pointer to data to send, s number of data
*/

//--------------------------------------------------------------------------
// Animated flame for Adafruit Pro Trinket.  Uses the following parts:
//   - Pro Trinket microcontroller (adafruit.com/product/2010 or 2000)
//     (#2010 = 3V/12MHz for longest battery life, but 5V/16MHz works OK)
//   - Charlieplex LED Matrix Driver (2946)
//   - Charlieplex LED Matrix (2947, 2948, 2972, 2973 or 2974)
//   - 350 mAh LiPoly battery (2750)
//   - LiPoly backpack (2124)
//   - SPDT Slide Switch (805)
//
// This is NOT good "learn from" code for the IS31FL3731; it is "squeeze
// every last byte from the Pro Trinket" code.  If you're starting out,
// download the Adafruit_IS31FL3731 and Adafruit_GFX libraries, which
// provide functions for drawing pixels, lines, etc.  This sketch also
// uses some ATmega-specific tricks and will not run as-is on other chips.
//--------------------------------------------------------------------------

#include "mbed.h"
#include "data.h"           // Flame animation data
//#define debug             // for some printf
#define I2C_ADDR 0xE8       // I2C address of Charlieplex matrix 0x74
I2C Wire(I2C_SDA, I2C_SCL);
uint8_t        page = 0;    // Front/back buffer control
const uint8_t *ptr  = anim; // Current pointer into animation data
uint8_t        img[9 * 16]; // Buffer for rendering image

//Buffer for I2C write method
char cmd[200];

// to select register on IS31FL3731, see datasheet page 9
void pageSelect(uint8_t n)
{
    cmd[0]=0xFD;
    cmd[1]=n;
    Wire.write(I2C_ADDR,cmd,2);  // attention write method close communications (stop condition is issued)
}

// SETUP FUNCTION - RUNS ONCE AT STARTUP -----------------------------------

void setup()
{
    uint8_t i, p, s;

    s=0;                // s is a counter for I2C buffer (cmd) write method
    pageSelect(0x0B);               // Access to the Function Registers (page Night)
    cmd[0]=0;            // adress first internal register 
    s++;                                       
    for(i=0; i<=0x0C; i++) 
    {
        cmd[i+1]=0x00;
        if (i==0) cmd[i+1]=0x00; //  0x08 for Auto Frame Play Mode   0x00 for picture mode
        if (i==0x0A) cmd[i+1]=0xFF;     // all to 0 except shutdown register
        s++;
    }
    Wire.write(I2C_ADDR,cmd,s);  

    for(p=0; p<2; p++) 
    {
        s=0;
        // For each page used (0 & 1)...
        pageSelect(p);                 // Access the Frame Registers
        cmd[0]=0;                      // Start from 1st LED control reg
        s++;
        for(i=0; i<0xB4; i++) 
        {
            if (i<18) cmd[i+1]=0xFF;          // Enable all LEDs (18*8=144)
            else cmd[i+1]=0x00;             // desable blink and PWM
            s++;
        }
        Wire.write(I2C_ADDR,cmd,s);
    }
}

void loop()
{
    uint8_t  a, x1, y1, x2, y2, x, y,s;
#ifdef debug
    printf("-------------------------------------------------------------\n");
    printf("start loop\n");
    printf("-------------------------------------------------------------\n");
#endif
    pageSelect(0x0B);    // Function registers
    cmd[0]=0x01; // Picture Display reg
    cmd[1]=page;
    Wire.write(I2C_ADDR,cmd,2);

    page ^= 1; // Flip front/back buffer index

    // Then render NEXT frame.  Start by getting bounding rect for new frame:
    a = *ptr++;     // New frame X1/Y1
    if(a >= 144)   // 0x90
    {
        // EOD marker? (valid X1 never exceeds 8)
        ptr = anim;                 // Reset animation data pointer to start
        a   = *ptr++; // and take first value
    }
    x1 = a >> 4;                  // X1 = high 4 bits
    y1 = a & 0x0F;                // Y1 = low 4 bits
    a  = *ptr++;                // New frame X2/Y2
    x2 = a >> 4;                  // X2 = high 4 bits
    y2 = a & 0x0F;                // Y2 = low 4 bits

    // Read rectangle of data from anim[] into portion of img[] buffer
    #ifdef debug
    printf("x1= %d y1= %d x2= %d y2 = %d \n",x1,y1,x2,y2);
    #endif
    for(x=x1; x<=x2; x++) {
        // Column-major
        for(y=y1; y<=y2; y++) 
        {
        img[(x << 4) + y] = *ptr++;
        #ifdef debug
        printf("x= %d y= %d data= %d \n",x,y,img[(x << 4) + y]);
        #endif
        }
    }

    // Write img[] to matrix (not actually displayed until next pass)
    pageSelect(page);    // Select background buffer
    s=0;
    cmd[0]=0x24; // First byte of PWM data
    s++;
    // copy img buffer to IS31FL3731
    for(uint8_t j=0; j<144; j++) 
    {
            cmd[j+1]=img[j];
            s++;
            #ifdef debug
            printf("j= %d img[j]= %d \n",j,img[j]);
            #endif
    }
    Wire.write(I2C_ADDR,cmd,s);
    wait_ms(30);            // can be adjuted for speed annimation
}

int main()
{
    #ifdef debug
    printf("-------------------------------------------------------------\n");
    printf("printf("Simulation flame with IS31FL3731\n\n");
    printf("-------------------------------------------------------------\n");
    #endif
    // arduino like
    setup();
    while(1) loop();
}
