#include "Wiegand.h"
#include "mbed.h"

Wiegand::Wiegand(PinName pdata0, PinName pdata1, PinName pHold, void (*onCardRead)()) :  _data0ISR(pdata0), _data1ISR(pdata1), _hold(pHold) {
    _onCardRead = onCardRead;
    /*
        stop reader from sending us any data
        and ensure after this block that no data through 
        wiegand gets sent
    */
    _hold = 1;
    wait_ms(250); 
    _sizeOfBuffer = sizeof(_buffer)/sizeof(*_buffer);
    _resetBuffer();
    
    /* Attach our interrupts, and start the ISRs */
    _attachInterrupts();
    _hold = 0; //start sending data
}

/*
    Check if we've read all card data
*/
void Wiegand::doEvents(void) {
    if(_bitsRead > 0 && _lastISR.read_ms() > 5) {
        _onCardRead();
        _resetBuffer();
        _hold = 0;   
    }
}
    
/*
    return number of bits currently read
*/
uint8_t Wiegand::bitsRead(void) {
    return _bitsRead;
}
    
/*
    Returns the bits read from in [start,end] (Inclusive).
    if [start,end] not in range of the bits read, 0 is returned
*/
uint64_t Wiegand::getBits(uint8_t start, uint8_t end) {
    //make sure the start and end indexes don't over index our read bit buffer
    if(!(start < _bitsRead) || !(end < _bitsRead)) {
        return 0;
    }
    uint64_t output = 0;
    for(int i = start; i <= end; ++i) {
        output <<=1;
        output |= 0x1 & _buffer[i]; 
    }
    return output;
}

/*
    Resets the buffer to its intial state (0), and _bitsRead = 0
*/
void Wiegand::_resetBuffer(void) {
     for(int i = 0, l = _sizeOfBuffer; i <l; ++i) {  _buffer[i] = 0; }
    _bitsRead = 0;
}

/*
    Attach ISRs to the rising edge
*/
void Wiegand::_attachInterrupts(void) {
    _data0ISR.rise(this, &Wiegand::_data0ISRAction);
    _data1ISR.rise(this, &Wiegand::_data1ISRAction);
}

/*
    Data0ISR Routine, called by rising edge of data0
*/
void Wiegand::_data0ISRAction(void) {
    _lastISR.reset();
    _lastISR.start();
    _hold = 1;
    if(_bitsRead < _sizeOfBuffer) {
        _buffer[_bitsRead++] = 0;
    }
}

/*
    Data1ISR Routine, called by rising edge of data1
*/
void Wiegand::_data1ISRAction(void) {
    _lastISR.reset();
    _lastISR.start();
    _hold = 1;
    if(_bitsRead < _sizeOfBuffer) {
        _buffer[_bitsRead++] = 1;
    }
}
