#ifndef TINYQ_H
#define TINYQ_H

/** \defgroup tinyQ Tiny Queue
 * Tiny Queues implements a 4 byte queue for iPhone events with just 1 byte of management overhead.
 * Head and Tail pointers both inherently wrap at the end of the buffer and no bounds checking is 
 * necesary IFF the buffer size is set as a power of 2 equal to the width of the pointers !!
 * i.e head is 2 bits wide so the buffer must be 2^2 = 4 bytes deep.
 * @{
 */
/** NOTE: Q_SIZE MUST be == 2 ^ widthof( tinyQ_t.head ) */
#define Q_SIZE 256

/** a tinyQ tracks a 8 * queue with just 1 byte overhead */
typedef struct
{
    unsigned char head:8;  ///< the bit depth MUST be that power of 2 that is the Q_SIZE
    unsigned char tail:8;  ///< the bit depth MUST be that power of 2 that is the Q_SIZE
    unsigned char fill:9;  ///< Must be 1 bit bigger than the head and tail pointers
    void* data[Q_SIZE];     ///< NOTE: Q_SIZE MUST be == 2 ^ widthof( tinyQ_t.head )
}tinyQ_t;


/**
 * \brief Push an Accessibility Event onto our queue for sending to the iphone
 * \param q pointer to our tinyQ_t
 * \param c the Event to send to the iphone
 * \return null if Q full,  1 on success
 * \sa Qpop(), Qjump()
 * \ingroup tinyQ
 */
int Qpush( tinyQ_t* q, void* p );

/**
 * \brief Push an Accessibility Event onto front of our queue for sending to the iphone
 * \param q pointer to our tinyQ_t
 * \param c the Event to send to the iphone
 * \return null if Q full, 1 on success
 * \sa Qpush(), Qpop()
 * \ingroup tinyQ
 */
int Qjump( tinyQ_t* q, void* p );

/**
 * \brief Pop an Accessibility Event off our queue for sending to the iphone
 * \param q pointer to our tinyQ_t
 * \param c pointer to receive the Event
 * \return null if Q empty, 1 on success
 * \sa Qpop(), Qjump()
 * \ingroup tinyQ
 */
int Qpop( tinyQ_t* q, void ** p);

#endif