//
// Main Functions for the TBR19 Sensor Health Checking Module
//
// Christopher Bull
//
// Student Number: 149034423
//
// Provides specific sensor health checking functions with values received from
// CAN bus reception and prints errors and warnings to both a locally stored
// ".txt" file and any connected USB PC terminal
// Also provides error and warning LED output pin signals for quick visual
// checking
//
// A configuration file for CAN IDs and sensor value adjustments is included 
// under "config.h"
//


// Include files and libraries required, also includes mbed library
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include "mbed.h"
#include "config.h"
#include "can_instruction_set.h"
#include "mcp_reg_addresses.h"


// Set up USB output and input for LPC1768 and PC, as well as digital output
// pins for the error and warning LEDS
Serial          pcHealth(USBTX, USBRX); // USB output and input
DigitalOut      warnLED(p21);           // Warning LED pin setup
DigitalOut      errorLED(p22);          // Warning LED pin setup


// Create local filesystem "logs" for logging data
LocalFileSystem local("logs");


// Function to reset the LEDs to the off status
void resetLEDs()
{
    // LEDs off before return
    warnLED = 0;
    errorLED = 0;
    return;
}


// Function to create and open the log file
void beginLog()
{
    // Open and create the text file created for logging, showing starting 
    // message at top of file to ensure system is logging
    FILE *filePtr = fopen("/logs/log.txt", "w");    // Open to write
    fprintf(filePtr, "This is the log file for the TBR19 sensor health "
                        "checking system.\n"
                        "Any errors or warnings will be printed here.\n\n"
                        "Begin log file:\n\n");     // Show start of log file
    fclose(filePtr);                                // Close file
    return;
}


// Checks for CAN error messages not created by the health checking system
void sensorErrorReceive(uint8_t* data)
{
    // Check to see if APPS implausibility has been detected and it is being
    // logged (this is an error)
    if(data[5] == APPS_IMPLAUS_ERR && APPS_IMPLAUS_HEALTH_ON == 1)
    {
        errorLED = 1;       // Turn on the error LED
        
        // Open the text file created for logging and write error to the file
        FILE *filePtr = fopen("/logs/log.txt", "a");    // Open to write
        fprintf(filePtr, "ERROR:   Accelerator Pedal Position Sensors values "
                            "are implausible!\n");      // Print error
        fclose(filePtr);                                // Close file
        
        pcHealth.printf("ERROR:   Accelerator Pedal Position Sensors values "
                            "are implausible!\n");      // Print to USB term
    } 
    
    return;    
}


// Checks to see if the APPS 1 value is outside normal bounds (results in error)
void checkRangeAPPS1(uint8_t* data)
{
    // Get APPS data and place into int form for calculation
    int appsData = data[5] << 8;
    appsData = appsData | data[6];
 
    // If outside either upper or lower bounds, trigger error protocol
    if(appsData > APPS_MAX || appsData < APPS_MIN)
    {
        errorLED = 1;       // Turn on error LED
        
        // Get error code and send along CAN bus under sensor error ID
        uint8_t appsRangeErr = APPS_1_RANGE_ERR;
        uint8_t* appsRangeErrPtr = &appsRangeErr;
        writeMessageCAN(SENSOR_ERROR_ID, appsRangeErrPtr, SENSOR_ERROR_DLC);
        
        // Open the text file created for logging and write error to the file
        FILE *filePtr = fopen("/logs/log.txt", "a");        // Open to write
        fprintf(filePtr, "ERROR:   Accelerator Pedal Position Sensor 1 value "
                            "is out of plausible range!\n");// Print error
        fclose(filePtr);                                    // Close file
        
        pcHealth.printf("ERROR:   Accelerator Pedal Position Sensor 1 value "
                            "is out of plausible range!\n");// Print to USB term
    }    
 
    return;
}


// Checks to see if the APPS 2 value is outside normal bounds (results in error)
void checkRangeAPPS2(uint8_t* data)
{
    // Get APPS data and place into int form for calculation
    int appsData = data[5] << 8;
    appsData = appsData | data[6];

    // If outside either upper or lower bounds, trigger error protocol
    if(appsData > APPS_MAX || appsData < APPS_MIN)
    {
        errorLED = 1;
        
        // Get error code and send along CAN bus under sensor error ID
        uint8_t appsRangeErr = APPS_2_RANGE_ERR;
        uint8_t* appsRangeErrPtr = &appsRangeErr;
        writeMessageCAN(SENSOR_ERROR_ID, appsRangeErrPtr, SENSOR_ERROR_DLC);

        // Open the text file created for logging and write error to the file
        FILE *filePtr = fopen("/logs/log.txt", "a");        // Open to write
        fprintf(filePtr, "ERROR:   Accelerator Pedal Position Sensor 2 value "
                            "is out of plausible range!\n");// Print error
        fclose(filePtr);                                    // Close file
        
        pcHealth.printf("ERROR:   Accelerator Pedal Position Sensor 2 value "
                            "is out of plausible range!\n");// Print to USB term
    }    
 
    return;
}


// Checks to see if the BSEF value is outside normal bounds (results in error)
void checkRangeBSEF(uint8_t* data)
{
    // Get BSE data and place into int form for calculation
    int bseData = data[5] << 8;
    bseData = bseData | data[6];

    // If outside either upper or lower bounds, trigger error protocol    
    if(bseData > BSE_F_MAX || bseData < BSE_F_MIN)
    {
        errorLED = 1;       // Turn on error LED
        
        // Get error code and send along CAN bus under sensor error ID
        uint8_t bseRangeErr = BSE_F_RANGE_ERR;
        uint8_t* bseRangeErrPtr = &bseRangeErr;
        writeMessageCAN(SENSOR_ERROR_ID, bseRangeErrPtr, SENSOR_ERROR_DLC);

        // Open the text file created for logging and write error to the file
        FILE *filePtr = fopen("/logs/log.txt", "a");    // Open to write
        fprintf(filePtr, "ERROR:   Front Brake System Encoder value is out of "
                            "plausible range!\n");      // Print error
        fclose(filePtr);                                // Close file
        
        pcHealth.printf("ERROR:   Front Brake System Encoder value is out of "
                            "plausible range!\n");      // Print to USB term
    }    
 
    return;
}


// Checks to see if the BSER value is outside normal bounds (results in error)
void checkRangeBSER(uint8_t* data)
{
    // Get BSE data and place into int form for calculation
    int bseData = data[5] << 8;
    bseData = bseData | data[6];
    
    // If outside either upper or lower bounds, trigger error protocol 
    if(bseData > BSE_R_MAX || bseData < BSE_R_MIN)
    {
        errorLED = 1;       // Turn on error LED
        
        // Get error code and send along CAN bus under sensor error ID
        uint8_t bseRangeErr = BSE_R_RANGE_ERR;
        uint8_t* bseRangeErrPtr = &bseRangeErr;
        writeMessageCAN(SENSOR_ERROR_ID, bseRangeErrPtr, SENSOR_ERROR_DLC);
        
        // Open the text file created for logging and write error to the file
        FILE *filePtr = fopen("/logs/log.txt", "a");    // Open to write
        fprintf(filePtr, "ERROR:   Rear Brake System Encoder value is out of "
                            "plausible range!\n");      // Print error
        fclose(filePtr);                                // Close file
        
        pcHealth.printf("ERROR:   Rear Brake System Encoder value is out of "
                            "plausible range!\n");      // Print to USB term
    }    
 
    return;
}


// Checks to see if front wheel speed is out of bounds and triggers warning is 
// this case
void checkRangeWheelSpeedF(uint8_t* data)
{
    // Get wheel speed data and place into float form for calculation
    float* wheelSpeedF;
    wheelSpeedF = reinterpret_cast<float*>(&data[5]);

    // If outside either upper or lower bounds, trigger warning protocol          
    if(*wheelSpeedF > WHEEL_SPEED_F_MAX)
    {
        warnLED = 1;        // Turn on warning LED
        
        // Get error code and send along CAN bus under sensor error ID
        uint8_t wsRangeErr = WHEEL_SPEED_F_RANGE_WARN;
        uint8_t* wsRangeErrPtr = &wsRangeErr;
        writeMessageCAN(SENSOR_ERROR_ID, wsRangeErrPtr, SENSOR_ERROR_DLC);
        
        // Open the text file created for logging and write error to the file
        FILE *filePtr = fopen("/logs/log.txt", "a");    // Open to write
        fprintf(filePtr, "WARNING: A front wheel speed sensor value is out of "
                            "plausible range!\n");      // Print error
        fclose(filePtr);                                // Close file
        
        pcHealth.printf("WARNING: A front wheel speed sensor value is out of "
                            "plausible range!\n");      // Print to USB term
    }    
 
    return;
}


// Checks to see if rear wheel speed is out of bounds and triggers warning is 
// this case
void checkRangeWheelSpeedR(uint8_t* data)
{
    // Get wheel speed data and place into float form for calculation
    float* wheelSpeedR;
    wheelSpeedR = reinterpret_cast<float*>(&data[5]);
           
    // If outside either upper or lower bounds, trigger warning protocol    
    if(*wheelSpeedR > WHEEL_SPEED_R_MAX)
    {
        warnLED = 1;        // Turn on warning LED
        
        // Get error code and send along CAN bus under sensor error ID
        uint8_t wsRangeErr = WHEEL_SPEED_R_RANGE_WARN;
        uint8_t* wsRangeErrPtr = &wsRangeErr;
        writeMessageCAN(SENSOR_ERROR_ID, wsRangeErrPtr, SENSOR_ERROR_DLC);
        
        // Open the text file created for logging and write error to the file
        FILE *filePtr = fopen("/logs/log.txt", "a");    // Open to write
        fprintf(filePtr, "WARNING: A rear wheel speed sensor value is out of "
                            "plausible range!\n");      // Print error
        fclose(filePtr);                                // Close file
        
        pcHealth.printf("WARNING: A rear wheel speed sensor value is out of "
                            "plausible range!\n");      // Print to USB term
    }    
 
    return;
}


// Checks to see if front dampers are out of bounds and gives a warning in
// this case
void checkRangeDamperF(uint8_t* data)
{
    // Get damper displacement data and place into int form for calculation
    int damperData = data[5] << 8;
    damperData = damperData | data[6];
    
    // If outside either upper or lower bounds, trigger warning protocol     
    if(damperData > DAMPER_POT_F_MAX || damperData < DAMPER_POT_F_MIN)
    {
        warnLED = 1;        // Turn on warning LED
        
        // Get error code and send along CAN bus under sensor error ID
        uint8_t damperRangeErr = DAMPER_POT_F_RANGE_WARN;
        uint8_t* damperRangeErrPtr = &damperRangeErr;
        writeMessageCAN(SENSOR_ERROR_ID, damperRangeErrPtr, SENSOR_ERROR_DLC);

        // Open the text file created for logging and write error to the file
        FILE *filePtr = fopen("/logs/log.txt", "a");        // Open to write
        fprintf(filePtr, "WARNING: A front damper displacement sensor value is "
                            "out of plausible range!\n");   // Print error
        fclose(filePtr);                                    // Close file
        
        pcHealth.printf("WARNING: A front damper displacement sensor value is "
                            "out of plausible range!\n");   // Print to USB term 
    }    
 
    return;
}


// Checks to see if rear dampers are out of bounds and gives a warning in
// this case
void checkRangeDamperR(uint8_t* data)
{
    // Get damper displacement data and place into int form for calculation
    int damperData = data[5] << 8;
    damperData = damperData | data[6];

    // If outside either upper or lower bounds, trigger warning protocol      
    if(damperData > DAMPER_POT_R_MAX || damperData < DAMPER_POT_R_MIN)
    {
        warnLED = 1;        // Turn on warning LED
        
        // Get error code and send along CAN bus under sensor error ID
        uint8_t damperRangeErr = DAMPER_POT_R_RANGE_WARN;
        uint8_t* damperRangeErrPtr = &damperRangeErr;
        writeMessageCAN(SENSOR_ERROR_ID, damperRangeErrPtr, SENSOR_ERROR_DLC);
        
        // Open the text file created for logging and write error to the file
        FILE *filePtr = fopen("/logs/log.txt", "a");        // Open to write
        fprintf(filePtr, "WARNING: A rear damper displacement sensor value is "
                            "out of plausible range!\n");   // Print error
        fclose(filePtr);                                    // Close file
        
        pcHealth.printf("WARNING: A rear damper displacement sensor value is "
                            "out of plausible range!\n");   // Print to USB term 
    }    
 
    return;
}


// Checks to see if the steering wheel hand clutch sensor has gone out of bounds
// which triggers an error protocol
void checkRangeHandClutch(uint8_t* data)
{
    // Get hand clutch displacement data and place into int form for calculation
    int clutchData = data[5] << 8;
    clutchData = clutchData | data[6];

    // If outside either upper or lower bounds, trigger error protocol     
    if(clutchData > HAND_CLUTCH_MAX || clutchData < HAND_CLUTCH_MIN)
    {
        errorLED = 1;       // Turn on error LED
        
        // Get error code and send along CAN bus under sensor error ID
        uint8_t clutchRangeErr = HAND_CLUTCH_RANGE_ERR;
        uint8_t* clutchRangeErrPtr = &clutchRangeErr;
        writeMessageCAN(SENSOR_ERROR_ID, clutchRangeErrPtr, SENSOR_ERROR_DLC);
    
        // Open the text file created for logging and write error to the file
        FILE *filePtr = fopen("/logs/log.txt", "a");        // Open to write
        fprintf(filePtr, "ERROR:   The steering wheel's hand clutch reading is "
                            "out of plausible range!\n");   // Print error
        fclose(filePtr);                                    // Close file
        
        pcHealth.printf("ERROR:   The steering wheel's hand clutch reading is "
                            "out of plausible range!\n");   // Print to USB term         
    }    
 
    return;
}


// Checks to see if steering angle out of bounds and gives a warning in
// this case
void checkRangeSteeringAngle(uint8_t* data)
{
    // Get steering angle data and place into int form for calculation
    int steeringData = data[5] << 8;
    steeringData = steeringData | data[6];

    // If outside either upper or lower bounds, trigger warning protocol 
    if(steeringData > STEERING_ANGLE_MAX || steeringData < STEERING_ANGLE_MIN)
    {
        warnLED = 1;        // Turn on warning LED
        uint8_t steeringRangeErr = STEERING_ANGLE_RANGE_WARN;
        uint8_t* steeringRangeErrPtr = &steeringRangeErr;
        writeMessageCAN(SENSOR_ERROR_ID, steeringRangeErrPtr, SENSOR_ERROR_DLC);
    
        // Open the text file created for logging and write error to the file
        FILE *filePtr = fopen("/logs/log.txt", "a");        // Open to write
        fprintf(filePtr, "WARNING: The steering angle position sensor value is "
                            "out of plausible range!\n");   // Print error
        fclose(filePtr);                                    // Close file
        
        pcHealth.printf("WARNING: The steering angle position sensor value is "
                            "out of plausible range!\n");   // Print to USB term 
    }    
 
    return;
}


// Checks for implausibility in front wheel speed values, results in warning
// Should be noted that a moderate difference in wheel speed is possible with
// turning and slip
void checkImplausibilityWheelSpeedF(float wheelSpeedFL, float wheelSpeedFR)
{
    // If either wheel speed is larger than the other by the max implausibility
    // then trigger warning protocol
    if(wheelSpeedFL - wheelSpeedFR > WHEEL_SPEED_F_IMPLAUS ||
        wheelSpeedFR - wheelSpeedFL > WHEEL_SPEED_F_IMPLAUS)
    {
        warnLED = 1;        // Turn on warning LED
        
        // Get error code and send along CAN bus under sensor error ID
        uint8_t wsImplausErr = WHEEL_SPEED_F_IMPLAUS_WARN;
        uint8_t* wsImplausErrPtr = &wsImplausErr;
        writeMessageCAN(SENSOR_ERROR_ID, wsImplausErrPtr, SENSOR_ERROR_DLC);
        
        // Open the text file created for logging and write error to the file
        FILE *filePtr = fopen("/logs/log.txt", "a");    // Open to write
        fprintf(filePtr, "WARNING: The front wheel speed sensor values are "
                            "implausible!\n");          // Print error
        fclose(filePtr);                                // Close file

        pcHealth.printf("WARNING: The front wheel speed sensor values are "
                            "implausible!\n");          // Print to USB term       
    }

    return;
}


// Checks for implausibility in rear wheel speed values, results in warning
void checkImplausibilityWheelSpeedR(float wheelSpeedRL, float wheelSpeedRR)
{
    // If either wheel speed is larger than the other by the max implausibility
    // then trigger warning protocol
    if(wheelSpeedRL - wheelSpeedRR > WHEEL_SPEED_R_IMPLAUS ||
        wheelSpeedRR - wheelSpeedRL > WHEEL_SPEED_R_IMPLAUS)
    {
        warnLED = 1;        // Turn on warning LED
        
        // Get error code and send along CAN bus under sensor error ID
        uint8_t wsImplausErr = WHEEL_SPEED_R_IMPLAUS_WARN;
        uint8_t* wsImplausErrPtr = &wsImplausErr;
        writeMessageCAN(SENSOR_ERROR_ID, wsImplausErrPtr, SENSOR_ERROR_DLC);
        
        // Open the text file created for logging and write error to the file
        FILE *filePtr = fopen("/logs/log.txt", "a");    // Open to write
        fprintf(filePtr, "WARNING: The rear wheel speed sensor values are "
                            "implausible!\n");          // Print error
        fclose(filePtr);                                // Close file
        
        pcHealth.printf("WARNING: The rear wheel speed sensor values are "
                            "implausible!\n");          // Print to USB term
    }

    return; 
}


