/*
    OSCtoCV Converter Configulation
*/

#pragma O3
#pragma Otime

#ifndef OSCtoCV_H
#define OSCtoCV_H

#include "mbed.h"
#include "FastIO.h" // https://developer.mbed.org/users/Sissors/code/FastIO/
//#include "FastAnalogIn.h"
#include "DebouncedInterrupt.h" // https://developer.mbed.org/users/kandangath/code/DebouncedInterrupt/
#include "BurstSPI.h" // https://developer.mbed.org/users/Sissors/code/BurstSPI/
#include "TextLCD.h"       //edit "writeCommand" "writeData" protected -> public
#include "EthernetNetIf.h"
#include "UDPSocket.h"
#include "OSCReceiver.h"
#include "mbedOSC.h"
#include "MIDI.h" // https://developer.mbed.org/users/okini3939/code/MIDI/

//-------------------------------------------------------------
// DAC8568 Control Bits (See datasheet)

#define WRITE               0x00
#define UPDATE              0x01
#define WRITE_UPDATE_ALL    0x02     // LDAC Write to Selected Update All
#define WRITE_UPDATE_N      0x03     // LDAC Write to Selected Update Respective
#define POWER               0x04
#define CLR                 0x05     // Clear Code Register
#define WRITE_LDAC_REG      0x06
#define RESET               0x07     // Software Reset DAC8568
#define SETUP_INTERNAL_REF  0x08

#define SPI_RATE            20000000 // 20Mbps SPI Clock
#define POLLING_INTERVAL    19       // Polling Interval (us)

//-------------------------------------------------------------
// OSCtoCV Converter Macros

#define Lin                 0        // Linear LinearCV
#define Chr                 1        // Chromatic
#define Maj                 2        // Major
#define M7                  3        // Major7
#define Min7                4        // Minor7
#define Dor                 5        // Dorian
#define Min                 6        // Minor
#define S5th                7        // 5th
#define Wht                 8        // Wholetone

#define SCALE_TOTAL         9        // Scales total

#define SCALE_AOUT          (65535 / SCALE_TOTAL - 1)

#define QUAN_RES1           116      // Quantize voltage Steps
#define QUAN_RES2           68
#define QUAN_RES3           46
#define QUAN_RES4           40
#define QUAN_RES5           68
#define QUAN_RES6           68  
#define QUAN_RES7           16
#define QUAN_RES8           58

#define SCALING_N       32256.0f

//------------------------------------------------------------- 
// Ethernet Setting

#define DHCP                   // address assigned by DHCP
#define INPUT_PORT      12345    // Input Port Number
#define TOSC_PORT        9000    // touchOSC Port Number

//------------------------------------------------------------- 
// SubModeCount
#define SUBMODE1_TOTAL      8
#define SUBMODE2_TOTAL      8

//------------------------------------------------------------- 
// Gate Sequencer Macros

#define _DISABLE            0
#define _ENABLE             1

#define GATE1               0
#define GATE2               1
#define GATE3               2
#define GATE4               3
#define SUBGATE             4
#define GATE_TOTAL          5

#define INVERT              1
#define NON_INVERT          0

#define GATESOUT_ON         0
#define GATESOUT_OFF        1

#define SYNC_ON             0
#define SYNC_OFF            1

//------------------------------------------------------------- 
// Beats (Note values)

#define N1ST                1   // whole
#define N2ND                2   // harf
#define N4TH                4   // quarter
#define N8TH                8
#define N16TH              16
#define N32TH              32
#define N64TH              64
#define NDOT2               3   // dotted
#define NDOT4               7
#define NDOT8               9
#define NDOT16             11
#define NDOT32             13
#define TRIP2               3   // triplets
#define TRIP4               6
#define TRIP8              12
#define TRIP16             24
#define TRIP32             48
#define SYNC24             96
#define NRESET              0   // Gate Reset

//------------------------------------------------------------- 
// Sequencer Macros
#define SHIFTSEQ            0
#define M185SEQ             1
#define EUCLID              2
#define BEATSSEQ            3

#define CV_CHANNEL1         1
#define CV_CHANNEL2         2
#define CV_CHANNEL3         3
#define CV_CHANNEL4         4
#define CV_CHANNEL5         5
#define CV_CHANNEL6         6
#define CV_CHANNEL7         7
#define CV_CHANNEL8         8

#define GATE_MODE_ADDRESS      "/gatemode"   // /gatemode1 ~ gatemode16
#define STEP_INDICATOR_ADDRESS "/seqstep/"   // touchOSC multi toggle(1x16(8)) for Current Step Indicator
#define RESET_COUNTER_ADDRESS  "/reset"      // touchOSC label for Sequencer reset count

#define ANALOG_JITTER       (rand() % 70 - 35)

//------------------------------------------------------------- 
// M185 Macros

#define PULSE_COUNT_ADDRESS       "/pulse"      // /pulse1 ~ pulse8        M185 Pulse Count
#define GATE185_MODE_ADDRESS      "/185gmode"   // /gatemode1 ~ gatemode8  M185 Gate Mode
#define STEP185_INDICATOR_ADDRESS "/185step/"   // touchOSC multi toggle(1x16(8)) for Current Step Indicator
#define RESET185_COUNTER_ADDRESS  "/185reset"   // touchOSC label for Sequencer reset count

#define SINGLE              0
#define MUTE                1
#define MULTI               2
#define HOLD                3

//------------------------------------------------------------- 
// Beats Sequencer Macros

#define BEATS_MATRIX_ADDRESS      "/bm/"     // touchOSC multi toggle(16x16) OSC address
#define BEATS_PULSE_COUNT_ADDRESS "/bp"      // /pulse1 ~ pulse16  BeatsSeq Pulse Count
#define BEATS_INDICATOR_ADDRESS   "/bps/"    // touchOSC multi toggle(1x16(8)) for Current Step Indicator
#define BEATS_COUNTER_ADDRESS     "/bprst"   // touchOSC label for Sequencer reset count

//------------------------------------------------------------- 
// Euclidean Sequencer Macros

#define READ_DELAY         10 // for debouncing 
#define MAXCHANNELS         4
#define MAXSTEPS           16 // max step length
#define TRIGGER_DURATION 2200

#define DISPLAY_UPDATE   2000 // how long active channel display is shown
#define MATRIX_ADDRESS   "/matrix/" // touchOSC multi toggle(9x16) OSC address


//-------------------------------------------------------------
// Global Variables

// Silentway Calibration Data Mapping
// http://www.expert-sleepers.co.uk/silentway.html

//  Chromatic Scale
extern const float calibMap1[QUAN_RES1];

//  Major Scale
extern const float calibMap2[QUAN_RES2];

//  M7(9)
extern const float calibMap3[QUAN_RES3];

//  m7(9)
extern const float calibMap4[QUAN_RES4];

//  Dorian Scale
extern const float calibMap5[QUAN_RES5];

//  Minor Scale
extern const float calibMap6[QUAN_RES6];

//  5th
extern const float calibMap7[QUAN_RES7];

//  Whole tone
extern const float calibMap8[QUAN_RES8];
//-------------------------------------------------------------

// Control Voltage
extern float gOSC_cv[8];
extern float gSeq_cv[16];
extern float g185_cv[8];

// Glide
extern float gGlide;

// Sub Mode
extern int gSubModeCount1;
extern int gSubModeCount2;

// Variables for Sequencer
extern float gPulseCount[8];
extern float gGateMode[16];
extern float gGateMode185[8];
extern float gSlide[16];
extern float gSlide185[8];
extern float gAccent[16];
extern float gAccent185[8];

// Variables for Beats Seq 
extern float gPulseCountBeats[16];
extern float gBeatsLevel[8];
extern float gBeatsDecay[8];
extern unsigned int gBeatsMatrix[8][16];

// Euclidean SEQ Variables
extern float gEucA[6], gEucB[6];

// Variables for Control
/*
gCtrl[0] /ctrl1 BPM
gCtrl[1] /ctrl2 Quantize mode
gCtrl[3] /ctrl4 Glide
gCtrl[4] /ctrl5 Shift CV Seq Reset Count
gCtrl[5] /ctrl6 M185 Seq Reset Count
gCtrl[6] /ctrl7 Gate Length
gCtrl[7] /ctrl8 Beats Seq Reset Count

gCtrlSW[0] /ctrlsw1 Sequencer STOP
gCtrlSW[1] /ctrlsw2 Euclidean Sequencer reset
gCtrlSW[2] /ctrlsw3 Sequencer Loop
gCtrlSW[3] /ctrlsw4 Euclid Seq ON
gCtrlSW[4] /ctrlsw5 ASR Analog Mode
gCtrlSW[5] /ctrlsw6 Beats Sequencer Random Vel

gArdPot[0] /pot1  Arduino pot1 control for Random, LFO
gArdPot[1] /pot2  Arduino pot2 control for Random, LFO
gArdSW[0] /sw1    Arduino sw1  control for Sub Mode1
gArdSW[1] /sw2    Arduino sw2  control for Sub Mode2
*/

extern float gCtrl[8];
extern bool  gCtrlSW[8];

// Variables for Arduino OSC
extern uint16_t gArdCV[4];
extern float gArdPot[2];
extern bool gArdSW[2];


//-------------------------------------------------------------
// mbed Functions

extern TextLCD      gLCD;   // rs, e, d4-d7

extern BurstSPI     gSPI;   // SPI (p6 unconnected)

extern FastOut<p15> gSYNCMODE;  // SYNC DAC8568
extern FastOut<p16> gLDAC;      // LDAC DAC8568

// GateOut
extern DigitalOut   gGATES[4];  // GateOut
extern FastOut<p19> gSUBGATE;   // SubGateOut
extern FastOut<p25> gCLOCKOUT;  // ClockOut

extern AnalogOut    gAOUT;

extern AnalogIn     gAIN;
extern DebouncedInterrupt  gSW; // Mode SW

// MIDI OUT
extern MIDI midi;

extern Timer        gTimer;     // Timer
extern Ticker       gPoller;    // Ticker for Polling

// Ethernet
extern EthernetNetIf    gEth;

// touchOSC Address
extern uint8_t touchOSCAddress[];
extern int touchOSCPort;

// Set OSC message for sending 
extern OSCClass osc;
extern OSCMessage sendMes;


//-------------------------------------------------------------
// Functions

/* Map Function 
Remaps float value from one range to another range.

float MapFloat(float x, float in_min, float in_max, float out_min, float out_max)

*/
extern inline float MapFloat(float x, float in_min, float in_max, float out_min, float out_max)
{
  return (x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
}

// Check current quantize scale
extern inline uint8_t CheckQuantizeMode(void)
{
    static uint8_t _qmode;
    uint8_t qmode;
    
    qmode = gCtrl[1] * (SCALE_TOTAL - 1);  // Sequencer Quantize Mode (gCtrl[1])

    if (_qmode != qmode)
    {
        gAOUT.write_u16(SCALE_AOUT * qmode); // write Analog out for send current scale
        _qmode = qmode;
    }
    
    return qmode;
}

// Fast strlen function http://www.strchr.com/optimized_strlen_function
extern inline size_t strlength(const char *s)
{
    size_t len = 0;
    
    for (;;) 
    {
        unsigned x = *(unsigned*)s;
        if ((x & 0xFF) == 0) return len;
        if ((x & 0xFF00) == 0) return len + 1;
        if ((x & 0xFF0000) == 0) return len + 2;
        if ((x & 0xFF000000) == 0) return len + 3;
        s += 4, len += 4;
    }
}

// set touchOSC Matrix Address 
extern inline char * SetMatrixAddress(int row, int column, int mode) 
{
    static char address[32];
    
    switch (mode)
    {
        case SHIFTSEQ:
        
            sprintf(address, "%s%d/1", STEP_INDICATOR_ADDRESS, column + 1);
            break;
        
        case M185SEQ:
        
            sprintf(address, "%s%d/1", STEP185_INDICATOR_ADDRESS, column + 1);
            break;
        
        case BEATSSEQ:
        
            sprintf(address, "%s%d/1", BEATS_INDICATOR_ADDRESS, column + 1);
            break;
            
        case EUCLID:
        
            sprintf(address, "%s%d/%d", MATRIX_ADDRESS, column + 1, row + 1);
            break;
    }
    
    return address;
    
/*
    static char address[32];
    char col[2];
    char ch[2];
    
    if(euclid)
    {
        strcpy(address, MATRIX_ADDRESS);
    
    } else {
        
        strcpy(address, STEP_INDICATOR_ADDRESS);
    }
    
    sprintf(col, "%d", column + 1);
    strcat(address, col);
    
    if(euclid)
    {
        strcat(address, "/");
        
        sprintf(ch, "%d", row + 1);
        strcat(address, ch);
    
    } else {
        
        strcat(address, "/1");
    }
    
    return address;
*/
}

// SPI Transfer
// DAC8568 data word length 32bit (8bit shift out)
extern void UpdateCV(unsigned int control, unsigned int ch, const unsigned int *data);

// Update CV Meter(Text LCD bar meter)
extern void UpdateCVMeter(unsigned int ch, const unsigned int *level);

// Check subMode1(gArdSW[0])
extern int CheckSubMode1(void);

// Check subMode2(gArdSW[1])
extern int CheckSubMode2(void);

// Update subMode Text LCD
extern void UpdateSubModeLCD(const char * subModeName);
    
#endif
