//
// Signal Generator DAC Driver
//
// Derived from AN10917: Memory to DAC data transfers using the LPC1700's DMA
//
// 
#ifndef SIGNALGENDAC_H
#define SIGNALGENDAC_H

#include "mbed.h"

#include "SignalGenDefs.h"       // access the waveform mode data type


#define SIGNAL_MEM_ENTRIES 2048     // size of the DAC buffer

/// The Signal Generator DAC Driver
///
/// This class provides the interface to first configure the DAC hardware characteristics,
/// and then to define and control the DAC output.
///
/// A choice of waveforms is available (Sine, Square, Triangle, Sawtooth, and User Defined.
///
/// @todo add support for User Defined waveform.
///
/// @code
/// SignalGenDAC g_signal;            // defaults to LPC1768 mbed module (p18 and 3.3v)
/// 
/// g_signal.PrepareWaveform(SG_SINE, 1000, 50, 2.2, 1.5);
/// g_signal.Start();
/// wait_ms(1000);
/// g_signal.Stop();
/// @endcode
///
class SignalGenDAC {

public:

    /// Constructor, which is used to define the hardware
    /// 
    /// The default parameters are based on the mbed LPC1768 micro, which has
    /// AnalogOut on p18 and uses a 3.3v supply for the A/D reference.
    ///
    /// @param[in] aout is the analog output pin
    /// @param[in] minV is based on the A/D low reference voltage (default 0.0)
    /// @param[in] maxV is based on the A/D high reference voltage (default 3.3)
    ///
    SignalGenDAC(PinName aout = p18, float minV = 0.0, float maxV = 3.3);

    /// Destructor
    ///
    ~SignalGenDAC();

    /// Create the waveform in the private memory buffer that is used to DMA to the DAC
    ///
    /// @param[in] mode defines the waveform: Sine, Square, Triangle, Sawtooth, User
    /// @param[in] frequency defines the desired frequency
    /// @param[in] dutycycle defined the duty cycle of the waveform to be created. The value
    ///             is range limited to 5 to 95 (representing 5 to 95 %).
    /// @param[in] voltage is the peak-to-peak voltage, and it range limited to 0 to 3.0.
    /// @param[in] offset is the offset voltage, and is range limited to 0 to 3.0.
    ///
    void PrepareWaveform(SG_Waveform mode, float frequency, float dutycycle, float voltage, float offset);

    /// Start the signal, in either a oneshot, or continuous mode.
    ///
    /// @param[in] oneShot defaults false, which causes continuous mode. 
    ///             When set true, one cycle is produced.
    ///
    void Start(bool oneShot = false);

    /// Stop the signal, if it is running.
    ///
    void Stop(void);

    /// Determine if the signal is running.
    ///
    /// @returns true if the signal is running.
    ///
    bool isRunning(void) { return isOn; }

private:
    bool isOn;              // tracks whether the signal is on or off
    AnalogOut * aout;
    float frequency;        // signal parameters
    float dutycycle;
    float voltage;
    float offset;
    float minV;             // Based on the A/D hardware
    float maxV;             // Based on the A/D hardware
    /// range limit a value.
    float rangelimit(float value, float min, float max);
    int numSamples;         // private container for number of samples
};

#endif // SIGNALGENDAC_H