/**
 * @file pcd8544_drv.hpp
 * PCD8544 Driver for an LCD with an PCD8544 controller.
 * Note: Text mode supported only.
 *
 * Created on: 12/31/2010 at 19:48
 * @author: CarlosFTM
 */
#ifndef __PCD8544_DRV_HPP__
#define __PCD8544_DRV_HPP__

#include "mbed.h"
#include "ascii_table.hpp"

/* Number of pixels on the LCD */
#define MAX_PIX_X  (84)
#define MAX_PIX_Y  (48)

/* Size of a caracter displayed on the LCD */
#define CHAR_PIX_X (6)
#define CHAR_PIX_Y (8)

/* Maximum number of character displayed on the LCD*/
#define MAX_CHAR_X (MAX_PIX_X / CHAR_PIX_X)
#define MAX_CHAR_Y (MAX_PIX_Y / CHAR_PIX_Y)

/* Timming constants*/
#define TICK        (1) // Clock pulse width in useg
#define RESET_TICKS (8) // number of ticks for reset

/* Digital output assignament*/
#define pin_sclk    (p21)
#define pin_mosi    (p22)
#define pin_dc      (p23)
#define pin_sce     (p24)
#define pin_reset   (p25)

/* Digital output logicla level*/
#define HIGH (1)
#define LOW  (0)

/* Display control command */
#define DISPLAY_BLANK (0x08)
#define NORMAL_MODE   (0x0C)
#define ALL_SEG_ON    (0x09)
#define INVERSE_MODE  (0x0D)

#define SET_ADDRES_X  (0x80)
#define SET_ADDRES_Y  (0x40)


/** Creates an interface to the PCD8544 LCD controller
 *
 */
class pcd8544
{
  private:
    DigitalOut _sclk;
    DigitalOut _mosi;
    DigitalOut _dc;
    DigitalOut _sce;
    DigitalOut _reset;
    /** Sends a raw byte to the LCD controller
    *
    * @param byte to send, boolean to indicate if it is a coomand (Command = TRUE / Data = FALSE).
    */
    void sendByte(char byte, bool command);
    
    /** Generates a clock tick
    *
    * @param pulse width in useg
    */
    inline void clockTick(unsigned short useg);

  public:
  
    /** Creates an interface to the LCD
    *
    * @param PinName for SCLK
    * @param PinName for MOSI
    * @param PinName for DC
    * @param PinName for SCE
    * @param PinName for RESET
    */
    pcd8544(PinName pin_sclk, PinName pin_mosi, PinName pin_dc, PinName pin_sce, PinName pin_reset);

    /** Writes data into the LCD controller
    *
    * @param data to be transmited in byte
    */
    void writeData(char data);

    /** Writes a command into the LCD controller
    *
    * @param command in byte
    */
    void writeCmd(char cmd);

    /** Writes a character into the LCD controller
    *
    * @param Character to be written in byte
    */
    void writeChar(char character);

    /** Writes a string into LCD
    *
    * @param pointer to a character string
    */
    void writeString(char* character);

    /** Reset the LCD
    *
    * @param none
    */
    void resetLCD(void);

    /** Initialize the LCD
    *
    * @param none
    */
    void initLCD(void);

    /** Clear the LCD screen
    *
    * @param none
    */
    void clearLCD(void);

    /** Set the cursor to a specific position
    *
    * @param X and Y position in characters
    */
    void setCursorXY(char x, char y);
};
#endif