#include "mbed.h"

Serial pc(USBTX,USBRX);
BusOut myled(LED1,LED2,LED3,LED4); // all leds of mbed lsb -> msb
DigitalOut ssel(p24);   // SSEL pio1_11/mbed p24
SPI spi(p5, p6, p7);    // mosi, miso, sclk

BusOut rgb(p26,p18,p25); // red, blue, green
InterruptIn button(p12); // 0= active/pushed 

// rgb buss assignments
#define RGB_RED   0x01
#define RGB_BLUE  0x02
#define RGB_GREEN 0x04

/*  7 Segment Display - this Rev A layout is rotated 180 degrees such that the
    decimal point is at the Top/Left

    Rev A schematic defines segent order as: 
    
    msb 7  6  5  4  3  2  1  0 
        F  C  .  B  A  G  E  D
        
    but setting individual segments shows:
    msb 7  6  5  4  3  2  1  0 
        F  .  E  D  G  B  A  C
  
                
             --A--
            |     |
            F     B
            |     |
             --G--
            |     |
            E     C
            |     |
             --D--
 */

// actual segments
#define SA  0xfd 
#define SB  0xfb 
#define SC  0xfe 
#define SD  0xef 
#define SE  0xdf 
#define SF  0x7f 
#define SG  0xf7 
#define SDP 0xbf 


#define WTIME   2.5     // delay between cycles

short curTest = 0;      // 0= all segments, 1= march segments 

/*      nextTest    ** Interrupt Event for pin fall **

    Advance curTest
*/

void nextTest(void) {
    curTest++;
    if (curTest >= 3)
      curTest = 0;
}

/*  setIO

    This procedure is used to set the following I/P ports:
    
    ph      myLed   Shows the test phase
    sv      SPI     7-Segment value 0=On
    lc      RGB     Led Color
*/    
void setIO(short ph, short sv, short lc) {
    myled = ph;         // identify phase

    ssel = 0;           // start ssel (sets RCLK to 0)    
    spi.write(sv);      // write to the shift register
    ssel = 1;           // Toggle RCLK/STCP to apply the SR --> NumDisplay (Rising edge action)

    rgb = lc;           // set led color
    wait(WTIME);        // common delay
}

/*      groupSegs

    This test groups all segments (G:A and DP)
    
    1:  Segs=Off, DP=Off, Red
    2:  Segs=Off, DP=On, Green
    3:  Segs:On,  DP=Off, Blue
    4:  Segs=On,  DP=On, RGB
*/
void groupSegs(void) {

    setIO(1,0xff,RGB_RED);      // 1: all Off, Red
    setIO(2,0xbf,RGB_GREEN);    // 2: b6 decimal on, others OFF, Green
    setIO(4,0x40,RGB_BLUE);     // 3: b6 decimal off, others On, Blue
    setIO(8,0x00,RGB_RED | RGB_GREEN | RGB_BLUE); // 3: all on, all colors
}

/*      marchBits

    Marches by bit position: 0 -> 7  

    This test marches the bits Low to turn on each segment
    in the order of bit 0 --> bit 7
    
    The initail setting is phase F and all segments Off
*/
    
void marchBits(void) {
    setIO(0xf,0xff,0);    // 1: all Off
    setIO(0,0xfe,0);      // 2: b0
    setIO(1,0xfd,0);      // 3: b1
    setIO(2,0xfb,0);      // 4: b2
    setIO(3,0xf7,0);      // 1: b3
    setIO(4,0xef,0);      // 2: b4
    setIO(5,0xdf,0);      // 3: b5
    setIO(6,0xbf,0);      // 4: b6
    setIO(7,0x7f,0);      // 4: b7
}

/*  marchSegs

    Marches by Segments: A -> G, DP  

    This test marches the bits Low to turn on each segment
    in the order of the segments A -> G, DP
    
    The initail setting is phase F and all segments Off
*/

void marchSegs(void) {
    setIO(0xf,0xff,0);    // 1: all Off
    setIO(0,SA,0);      // 2: b1
    setIO(1,SB,0);      // 3: b2
    setIO(2,SC,0);      // 4: b0
    setIO(3,SD,0);      // 1: b4
    setIO(4,SE,0);      // 2: b5
    setIO(5,SF,0);      // 3: b7
    setIO(6,SG,0);      // 4: b3
    setIO(7,SDP,0);     // 4: b6
}


/*      main

*/
int main() {
    pc.printf("\nEA: Segment & Board I/O Tests\n");
    pc.printf("\n Depress/Push Joystick to change tests at end of test cycle\n");
    button.fall(&nextTest);
    
    myled = 0;
    spi.format(8,1);    // set format as 8b/pol:pha=0
    spi.frequency(1000000); // default is 1mhz

    /* curTest is advanced by the depression/push down of the Joystick */  
    while(1) {
     switch (curTest){
       case 0:
         printf("  Group Segments & RGB Led...\n");
         groupSegs(); 
         break;
       
       case 1:  
         printf("  March Bits b0 => b7...\n");
         marchBits(); 
         break;
         
       case 2:  
         printf("  March Segments SA => SG, DP...\n");
         marchSegs(); 
         break;
         
       default: curTest = 0; break;
     }
   }
}

    