/* WiFi Example
 * Copyright (c) 2018 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "TCPSocket.h"
#include "stm32l475e_iot01_tsensor.h"

#define FIRE_DETECTED (1UL << 0)

#define WIFI_IDW0XX1    2
#define TS_DEVICE "stmWifi"
#define thingspeak_APIkey_write "SCS2JFDTW4EZC0NP"
#define thingspeak_APIkey_read "O3ZVNRCX5J95EB4G"

#if (defined(TARGET_DISCO_L475VG_IOT01A) || defined(TARGET_DISCO_F413ZH))
#include "ISM43362Interface.h"
ISM43362Interface wifi(MBED_CONF_APP_WIFI_SPI_MOSI, MBED_CONF_APP_WIFI_SPI_MISO, MBED_CONF_APP_WIFI_SPI_SCLK, MBED_CONF_APP_WIFI_SPI_NSS, MBED_CONF_APP_WIFI_RESET, MBED_CONF_APP_WIFI_DATAREADY, MBED_CONF_APP_WIFI_WAKEUP, false);

#else // External WiFi modules
#if MBED_CONF_APP_WIFI_SHIELD == WIFI_IDW0XX1
#include "SpwfSAInterface.h"
SpwfSAInterface wifi(MBED_CONF_APP_WIFI_TX, MBED_CONF_APP_WIFI_RX);
#endif // MBED_CONF_APP_WIFI_SHIELD == WIFI_IDW0XX1
#endif

//LEDS used for debugging
DigitalOut  led1(LED1, 1);
DigitalOut  led2(LED2, 1);

//Buzzer
PwmOut buzzer(D9);

//Interruptions
InterruptIn button(D3);
InterruptIn IR(D7);

//Thread
Thread thread;

//Event flag to control thread start
EventFlags event_flags;



void button_ISP()
{
    led1 = 0;
    NVIC_SystemReset();
}

void IR_ISP()
{
    led1 = 1;
    event_flags.set(FIRE_DETECTED);
    IR.rise(NULL);
}

const char *sec2str(nsapi_security_t sec)
{
    switch (sec) {
        case NSAPI_SECURITY_NONE:
            return "None";
        case NSAPI_SECURITY_WEP:
            return "WEP";
        case NSAPI_SECURITY_WPA:
            return "WPA";
        case NSAPI_SECURITY_WPA2:
            return "WPA2";
        case NSAPI_SECURITY_WPA_WPA2:
            return "WPA/WPA2";
        case NSAPI_SECURITY_UNKNOWN:
        default:
            return "Unknown";
    }
}

void main_routine()
{
    //Temperature reading
    BSP_TSENSOR_Init();
    float temperature_value = 0;
    int fire_state=1;
    //Check fire variable
    int numReadings = 0;

    while(true) {
        //wait for a fire reading
        event_flags.wait_any(FIRE_DETECTED);

        //check against false readings
        for(int n = 0; n < 5; n++) {
            numReadings += IR.read();
            led1 = !led1;
            wait(0.2);
        }

        printf("\nNum of positive readings: %d\n\r", numReadings);

        if(numReadings >= 5) {
            buzzer.write(1.0f);
            printf("\nFIRE DETECTED - SENDING DATA\n\r");
            IR.rise(NULL);
            led1 = 1;
            led2 = 1;

            //Connect to ThingSpeak and starting sending data
            int ret = 1;
            do {
                printf("\nConnecting to %s...\n", MBED_CONF_APP_WIFI_SSID);
                ret = wifi.connect(MBED_CONF_APP_WIFI_SSID, MBED_CONF_APP_WIFI_PASSWORD, NSAPI_SECURITY_WPA_WPA2);
            } while(ret != 0);

            printf("Success\n\n");
            printf("MAC: %s\n", wifi.get_mac_address());
            printf("IP: %s\n", wifi.get_ip_address());
            printf("Netmask: %s\n", wifi.get_netmask());
            printf("Gateway: %s\n", wifi.get_gateway());
            printf("RSSI: %d\n\n", wifi.get_rssi());

            NetworkInterface *net = &wifi;

            TCPSocket socket;
            nsapi_error_t response;
            char sbuffer[256];
            char message[64];

            while(true) {
                
                buzzer.write(1.0f);
                // Open a socket on the network interface, and create a TCP connection to thingspeaks.com
                socket.open(net);
                response = socket.connect("api.thingspeak.com", 80);
                if(0 != response) {
                    printf("Error connecting: %d\n", response);
                    socket.close();
                    return;
                }

                printf("Connected to the Server\n");

                //lecture des données des capteurs et actualisation des variables à transmettre
                temperature_value = BSP_TSENSOR_ReadTemp() - 8;
                led1 = !led1;


                /* Construct content of HTTP command */
                //message à transmettre (données des capteurs)
                sprintf(message, "{\"field1\": %0.2f, \"field2\": %d}", temperature_value,fire_state);
                printf("Content Length = %d\r\n", (int)strlen(message));

                /* Construct HTTP command to send */
                // Phase de transmission des données à ThingSpeaks.com
                sprintf(sbuffer, "GET /update?api_key=%s HTTP/1.1\r\nHost: api.thingspeak.com\r\nContent-Type: application/json\r\nContent-Length: %d\r\n\r\n%s", thingspeak_APIkey_write,
                        (int)strlen(message),message);
                printf("HTTP command %s\r\n", sbuffer);
                wait(1.0); // temporisation avant la nouvelle transmission

                // Send a simple http request
                printf("Sending HTTP request to thingspeak.com...\n");//
                nsapi_size_t size = strlen(sbuffer);
                response = 0;
                while(size) {
                    response = socket.send(sbuffer+response, size);
                    if (response < 0) {
                        printf("Error sending data: %d\n", response);
                        socket.close();
                        return;
                    } else {
                        size -= response;
                        // Check if entire message was sent or not
                        printf("sent %d [%.*s]\n", response, strstr(sbuffer, "\r\n")-sbuffer, sbuffer);
                    }
                }
                 buzzer.write(0.8f);

                // Receive a simple http response and print out the response line
                char rbuffer[64];
                response = socket.recv(rbuffer, sizeof rbuffer);
                if (response < 0) {
                    printf("Error receiving data: %d\n", response);
                } else {
                    printf("recv %d [%.*s]\n", response, strstr(rbuffer, "\r\n")-rbuffer, rbuffer);
                }
                // Close the socket to return its memory and bring down the network interface
                socket.close();
               fire_state=0;
            }


        } else {
            numReadings = 0;
            if(IR.read() == 0) {
                event_flags.set(0);
                IR.rise(&IR_ISP);
                printf("\nNo fire detected, returning IDLE\n\r");
            } else {
                event_flags.set(FIRE_DETECTED);
                printf("\nSensor still high, re-checking\n\r");
            }
        }


    }
}



int main()
{
    buzzer.period(0.5f);      // 4 second period
    buzzer.write(0.0f);
    led1 = 0;
    led2 = 0;
    thread.start(mbed::callback(main_routine));
    button.fall(&button_ISP);
    IR.rise(&IR_ISP);
    //main function will sleep
    while(true) {
        wait(1);
    }

}
