#include "mbed.h"
#include "MODSERIAL.h"
#include "SDFileSystem.h"


////////////////////Declarations////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

// Thermocouple pin declarations
DigitalOut SCK_1(PTD1);       // sck pin of thermocouples
DigitalOut CS_1(PTE25);        // chip select pin of thermocouples


//DigitalIn pins[12]={D2, D3, D4, D5, D6, D7, D8, D9, D10, D11, D12, D13};        //thermocouples 1 - 12 select corresponding pin numbers mbed
DigitalIn pins[3]={D2, D3, D4};        //thermocouple select corresponding pin numbers mbed


// Torque sensor
AnalogIn fcs(PTB2);     // force sensor output connected to analog 0


// LED
DigitalOut r_led(LED_RED);
DigitalOut g_led(LED_GREEN);
DigitalOut b_led(LED_BLUE);

// Serial connection and SD
MODSERIAL pc(USBTX,USBRX, 1024, 512);
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd"); // MOSI, MISO, SCK, CS
FILE *fp;

// Other declarations
Ticker tick;

const int baudrate = 115200;      // baud rate
const float Ts = 0.25;            // sample time (sec)

const int led_on = 0;
const int led_off = 1;

volatile bool fn1_go = 0;                // go-flag starts in disabled state
volatile bool start = 0;
volatile bool meas = 0;

float fcs_read;

int trial_numb = 0;
int cal_numb = 1;

char filename_torque[64];
char filename_temp[64];
char filename_cal[64];




////////////////////Functions///////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

void led_control(bool r, bool g, bool b)    // control rgb LEDs
{
    r_led.write(r);
    g_led.write(g);
    b_led.write(b);
    return;
}

void fn1_activate()     // function called by ticker every Ts seconds
{
    fn1_go = 1;      // go-flag active
}


void readTC()
{
    //uses MAX31855 chip
    //see https://datasheets.maximintegrated.com/en/ds/MAX31855.pdf for documentation
    
    
    float d = 0;
    int16_t b[3] = {0,0,0};

    CS_1 = 0;       //Active-Low Chip Select. Set CS low to enable the serial interface.
    wait_us(0.2);
    
    for (int i = 15; i >= 0; i--) {
        SCK_1 = 0;                      //set clock value to 0
        wait_us(0.1);                   //wait for 100ns
        SCK_1 = 1;                      //then set clock high again
        wait_us(0.1);                   //wait for 100ns
        for (int j = 0; j < 3; j++) { //read out bits for each thermocouple (0 to 11)
            if (pins[j]) {
                b[j] |= (1 << i);       //bitshift
            }
        }
    }
    
    CS_1 = 1;
    
    for (int j = 0; j < 3; j++) {     //for every thermocouple (starting from 0 to 11, so 12 in total)
        if (b[j] & 0x1) {               //b[j] contains the bits that are received from every chip. if least significant bit is 1, then open circuit.
            b[j] = 20000;
        } else {
            b[j] &= 0xfffc;             //if no fault then select 14 bit thermocouple data from bitstring
            d = b[j] / 16.0;
        }
        //fprintf(fp, "%.3f \n\r", d);
        pc.printf("%.3f\t", d);
    }
    pc.printf("\n\r");
    //fprintf(fp, "\n\r");
    return;

}

void getsample(bool meas)
{
    if(meas){                                   // alleen data uitlezen en opslaan als meas = 1
        fcs_read = fcs.read();                  // capture torque sensor data   
        //fprintf(fp, "%.3f\t", fcs_read);        // output data to file
        pc.printf("%.3f\t", fcs_read);          // output data to screen
        readTC();                               // call function to read thermocouple data    
    }
    return;
}


////////////////////Main////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

int main()
{
    pc.baud(baudrate);
    tick.attach(&fn1_activate,Ts);
    led_control(led_off,led_off,led_off);   // start with LEDs off
    
    while (1) {
        if(fn1_go) {                        // statements to execute when go-flag is active
            fn1_go = 0;
            getsample(meas);                // deze functie wordt elke ts seconde aangeroepen. afhankelijk van waarde meas (0 of 1) wordt de data opgeslagen in textfile
        }

        if(pc.readable()) {                 // if character available
            switch(pc.getc()) {             // read a character from keyboard
                case 'c':
                    led_control(led_on,led_on,led_off);                         //orange LED during calibration
                    sprintf(filename_cal, "/sd/cal_torque_%d.txt", cal_numb);   //construct new textfile
                    fp = fopen(filename_cal, "w");
                    if(fp == NULL) {
                        error("Could not open file for write\n");
                    } else {
                        while(1) {
                            for (int i = 0; i<50; i++) {                    //get 50 samples from torque sensor
                                fcs_read = fcs.read();
                                fprintf(fp, "%.3f\t \n\r", fcs_read);       //output values to the screen
                            }
                            break;
                        }
                    }
                    fclose(fp);                                             //close file
                    cal_numb++;                                             //increment calibration number
                    pc.printf("File saved as %s\n\r", filename_cal);        //print name of calibration file to screen
                    pc.printf("Press 'c' to record new calibration or 't' to start new trial\n\r");    // print message to screen
                    break;
                case 't':
                    trial_numb++;
                    pc.printf("Trial number: %d\n\r", trial_numb);
                    pc.printf("Press 's' to start measurement\n\r");
                    break;
                case 's':
                    start = !start;
                    if (start) {
                        led_control(led_off,led_on,led_off);                        //green LED during measurement
                        sprintf(filename_torque, "/sd/trial%d.txt", trial_numb);    //construct new textfile to store data from torque sensor
                        fp = fopen(filename_torque, "w");
                        if(fp == NULL) {
                            error("Could not open file for write\n\r");
                        } else {
                            pc.printf("Measurement started... \n\r");
                            pc.printf("Press 's' to stop measurement\n\r");
                            meas = 1;                                               // hiermee zorg je dat de functie getsample de data wegschrijft naar het .txt file
                        }
                        break;
                    } else {
                        led_control(led_on,led_off,led_off);                                  // RED LED when ready to stopped measuring
                        meas = 0;                                                   // hiermee stop je de output naar het .txt file
                        fclose(fp);
                        pc.printf("File saved as %s\n\r", filename_torque);                                // print filename to screen
                        pc.printf("Press 'c' to perform new calibration or 't' to start new trial\n\r");    // print message to screen

                    }
                    break;
            }
        }
    }
}
