/* JKsoft Wallbot BLE Library
 *
 * wallbotble.h
 *
 * Copyright (c) 2010-2014 jksoft
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef WALLBOT_BLE_H
#define WALLBOT_BLE_H

#include "mbed.h"
#include "TB6612.h"
//#include "QEI.h"
#include "PID.h"

#define PulsesPerRev	24

#define Kp  0.0001
#define Ki  0.001
#define Kd  0.0
#define RATE 0.01 // 制御周期(sec) 0.01=10ms

/** wallbot ble control class
 *
 * Example:
 * @code
 * // Drive the wwallbot forward, turn left, back, turn right, at half speed for half a second

#include "mbed.h"
#include "wallbotble.h"

wallbotble wb;
 
int main() {

    wb.sensor_calibrate();

    while(!wb.GetSw())
    {
        wb.set_led(wb.GetLinePosition());
    }
    
    wb.forward(1.0);
    wait (1.0);
    wb.left(1.0);
    wait (1.0);
    wb.backward(1.0);
    wait (1.0);
    wb.right(1.0);
    wait (1.0);
    
    wb.stop();

    while(1);

 }

 * @endcode
 */
class wallbotble  {

    // Public functions
public:

    /** Create the wallbot object connected to the default pins
     */
    wallbotble();
    
    
    /** Sensor calibrate
     *
     */
    void f_sensor_calibrate (void);

    /** Directly control the speed and direction of the left motor
     *
     * @param speed A normalised number -1.0 - 1.0 represents the full range.
     */
    void left_motor (float duty);

    /** Directly control the speed and direction of the right motor
     *
     * @param speed A normalised number -1.0 - 1.0 represents the full range.
     */
    void right_motor (float duty);

    /** Drive both motors forward as the same speed
     *
     * @param speed A normalised number 0 - 1.0 represents the full range.
     */
    void forward (float duty);

    /** Drive both motors backward as the same speed
     *
     * @param speed A normalised number 0 - 1.0 represents the full range.
     */
    void backward (float duty);

    /** Drive left motor backwards and right motor forwards at the same speed to turn on the spot
     *
     * @param speed A normalised number 0 - 1.0 represents the full range.
     */
    void left_turn (float duty);

    /** Drive left motor forward and right motor backwards at the same speed to turn on the spot
     * @param speed A normalised number 0 - 1.0 represents the full range.
     */
    void right_turn (float duty);

    /** Stop both motors
     *
     */
    void stop (void);
	
    /** ラインの推定値を返す。 左端-1500～中央0～右端1500
     * 一定以上の検出がない場合は前回値から右端、左端と判定
     */	
	short GetLinePosition(void);
	
    /** Get switch .(switch OFF:0 or ON:1 return.)
     *
     */	
	int GetSw(void);
	
    /** Set status led .
     * @param led (bit0:LEFT bit2:RIGHT)
     */	
	void set_led(char bit);

    /** Set led1 .
     * @param led (0:off,1:on)
     */	
	void set_led1(char bit);

    /** Set led2 .
     * @param led (0:off,1:on)
     */	
	void set_led2(char bit);
	
	// RPM指令で左右モータを制御する。
	void SetRPM(float leftRPM, float rightRPM);    
		
	// キャリブレーションのリセット
	void resetCalibration();

	// 自動キャリブレーション
	void auto_calibrate();
	
	//ラインセンサ値を補正し0から1000にマッピングしてsensor_valuesにロードする。
	void readCalibrated();
	
	unsigned short sensor_values[4];  
	
	
	void control_enable(bool enable);	

	unsigned short _calibratedMinimum[4];  
	unsigned short _calibratedMaximum[4];  
	
	// 左右のモーター回転数を返す
	float get_right_rpm();
	float get_left_rpm();
	
    unsigned short _right_pulses;
    unsigned short _left_pulses;
    
private :
	//ff tearm
	float ff_r,ff_l;
	//後退用フラグ
	bool _right_back;
	bool _left_back;
	
	float _right_rpm;
	float _left_rpm;
	

	Timer _right_timer;
	Timer _left_timer;
	void right_count();
	void left_count();
	
	void tick_callback();
	
	Ticker _control_tic;

	TB6612 _right_motor;
	TB6612 _left_motor;
	
	BusIn _sw;
	DigitalOut _outlow;
	BusOut _statusled;
	AnalogIn _f_sensor1;
	AnalogIn _f_sensor2;
	AnalogIn _f_sensor3;
	AnalogIn _f_sensor4;
	int _sensor_gain;
	
    InterruptIn _right_enc;
    InterruptIn _left_enc;
    
    
	PID _ctrl_r;
	PID _ctrl_l;
};

#endif