#include "mbed.h"
#include "i2c_mbed_fpga.h"


/** i2c read from slave DUT
*  retun 0 on success, otherwise fails
*
*  @param i2c_master specifies the i2c interface
*  @param word is 4byte, first 2bytes as addr, the rest 2bytes to store data
*/
int i2c_mbed_fpga::i2c_word_read(char *word){
    int ack = 0;
    ack = i2c_master.write(I2C_SLAVE_ADDR, word, 1, true);  //restart
    ack += i2c_master.read(I2C_SLAVE_ADDR, word+1, 2, false); //stop bit
    wait_us(100); 
    return (ack == 0) ? 0 : 1; 
}

/** i2c write to slave DUT
*   ==> one time write, not read back check
*
*  @param i2c_master specifies the i2c interface
*  @param word is considered as 4byte char data
*/
int i2c_mbed_fpga::i2c_word_write(char *word){
    int ack = 0;
    ack = i2c_master.write(I2C_SLAVE_ADDR, word, 3, false);
    return ack; 
}


/** i2c enter key to open I2C window (for old releases)
*/
//int i2c_mbed_fpga::i2c_window_open(){
//    char i2cMessage[3];
//    *(i2cMessage+0) = (char)(I2C_CUST_ID3)& 0xff;
//    *(i2cMessage+1) = (char)(0xD0)& 0xff;
//    *(i2cMessage+2) = (char)(0xD0)& 0xff;  
//    return i2c_word_write(i2cMessage);
//}

/** i2c enter key to Start the motor (for old releases)
*/
//int i2c_mbed_fpga::i2c_motor_start(){
//    char i2cMessage[3];
//    *(i2cMessage+0) = (char)(I2C_CUST_ID3)& 0xff;
//    *(i2cMessage+1) = (char)(0xCA)& 0xff;
//    *(i2cMessage+2) = (char)(0xFE)& 0xff;  
//    return i2c_word_write(i2cMessage);
//}


/** i2c enter key to open I2C configuration mode entry
*/
int i2c_mbed_fpga::i2c_config_mode_entry(){
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_COMMAND_CONTROL)& 0xff;
    *(i2cMessage+1) = (char)(0x1D)& 0xff;
    *(i2cMessage+2) = (char)(0xEA)& 0xff;  
    return i2c_word_write(i2cMessage);
}

/** i2c enter MLX key to open I2C MLX configuration mode entry
*/
int i2c_mbed_fpga::i2c_mlx_mode_entry(){
    int ack = 0;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_COMMAND_CONTROL)& 0xff;
    *(i2cMessage+1) = (char)(0x35)& 0xff;
    *(i2cMessage+2) = (char)(0x4B)& 0xff;  
    ack += i2c_word_write(i2cMessage);
    *(i2cMessage+0) = (char)(I2C_COMMAND_KEY)& 0xff;
    *(i2cMessage+1) = (char)(0x65)& 0xff;
    *(i2cMessage+2) = (char)(0xA9)& 0xff;  
    ack += i2c_word_write(i2cMessage);
    
    return ack;
}


/** i2c ram start up flag set to skip OTP copy 
*/
int i2c_mbed_fpga::i2c_skip_app_copy(){
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_STARTUP_FLAGS_1)& 0xff;
    *(i2cMessage+1) = (char)(0x05)& 0xff;
    *(i2cMessage+2) = (char)(0x00)& 0xff;  
    return i2c_word_write(i2cMessage);
}

/** i2c soft reset 
*/
int i2c_mbed_fpga::i2c_soft_reset(){
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_COMMAND_CONTROL)& 0xff;
    *(i2cMessage+1) = (char)(0xC1)& 0xff;
    *(i2cMessage+2) = (char)(0xA0)& 0xff;  
    return i2c_word_write(i2cMessage);
}


/** i2c to set the 50k PWM
*/
int i2c_mbed_fpga::i2c_set_50k_pwm(unsigned int pwm50k){
    nv_gen_ctrl_val &= ~NV_PWM_50K_MASK;
    nv_gen_ctrl_val |= pwm50k << NV_PWM_50K_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_GEN_CTRL >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_gen_ctrl_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_gen_ctrl_val >> 0)& 0xff;          
    return i2c_word_write(i2cMessage);
}



/** i2c to set the Postion Pulse width
*/
int i2c_mbed_fpga::i2c_set_position_pulse_width(unsigned int mantisaa_2b, unsigned int exponent_3b){
    nv_positin_val &= ~NV_POSITION_PULSE_TIME_MASK;
    nv_positin_val |= ((exponent_3b << 2) | mantisaa_2b) << NV_POSITION_PULSE_TIME_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_POSITION >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_positin_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_positin_val >> 0)& 0xff;          
    return i2c_word_write(i2cMessage);
}

/** i2c to set the Postion Pulse duty cycle
*/
int i2c_mbed_fpga::i2c_set_position_duty(unsigned int duty_2b){
    nv_positin_val &= ~NV_POSITION_DUTY_MASK;
    nv_positin_val |= duty_2b << NV_POSITION_DUTY_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_POSITION >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_positin_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_positin_val >> 0)& 0xff;          
    return i2c_word_write(i2cMessage);
}

/** i2c to enable the Postion Pulse majority volting
*/
int i2c_mbed_fpga::i2c_set_position_maj_vote(unsigned int maj_1b){
    nv_positin_val &= ~NV_POSI_MAJO_VOTE_MASK;
    nv_positin_val |= maj_1b << NV_POSI_MAJO_VOTE_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_POSITION >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_positin_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_positin_val >> 0)& 0xff;          
    return i2c_word_write(i2cMessage);
}

/** i2c to set the anti-cogging rotation direction
*/
int i2c_mbed_fpga::i2c_set_position_anti_cog(unsigned int cog_1b){
    nv_positin_val &= ~NV_ANTI_COG_MASK;
    nv_positin_val |= cog_1b << NV_ANTI_COG_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_POSITION >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_positin_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_positin_val >> 0)& 0xff;          
    return i2c_word_write(i2cMessage);
}


/** i2c to set the Start Up Pulse width (pulse train)
*/
int i2c_mbed_fpga::i2c_set_start_up_pulse_width(unsigned int mantisaa_3b, unsigned int exponent_3b){
    nv_start_up_val &= ~NV_START_UP_TIME_MASK;
    nv_start_up_val |= ((exponent_3b << 3) | mantisaa_3b) << NV_START_UP_TIME_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_START_UP >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_start_up_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_start_up_val >> 0)& 0xff;          
    return i2c_word_write(i2cMessage);
}

/** i2c to set the Start up Pulse duty cycle (pulse train)
*/
int i2c_mbed_fpga::i2c_set_start_up_duty(unsigned int duty_2b){
    nv_start_up_val &= ~NV_START_DUTY_MASK;
    nv_start_up_val |= duty_2b << NV_START_DUTY_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_START_UP >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_start_up_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_start_up_val >> 0)& 0xff;          
    return i2c_word_write(i2cMessage);
}

/** i2c to set the Start up commutation number of EHPs (pulse train)
*/
int i2c_mbed_fpga::i2c_set_start_up_num_comm(unsigned int comm){
    nv_start_up_val &= ~NV_COMM_START_NUM_MASK;
    nv_start_up_val |= comm << NV_COMM_START_NUM_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_START_UP >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_start_up_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_start_up_val >> 0)& 0xff;          
    return i2c_word_write(i2cMessage);
}

/** i2c to set the Soft Start Up (pulse train)
*/
int i2c_mbed_fpga::i2c_set_soft_start_up(unsigned int enbale, unsigned int mantisaa_3b, unsigned int exponent_3b, unsigned int step_size, unsigned int num_steps){
    int ack = 0;
    nv_start_up_val &= ~NV_SOFT_START_MASK;
    nv_start_up_val |= enbale << NV_SOFT_START_OFFSET;
    nv_positin_val &= ~NV_FIRST_NON_FLAT_TIME_MASK;
    nv_positin_val |= ((exponent_3b << 3) | mantisaa_3b) << NV_FIRST_NON_FLAT_TIME_OFFSET;
    nv_start_up_val &= ~NV_SOFT_STEP_SIZE_MASK;
    nv_start_up_val |= step_size << NV_SOFT_STEP_SIZE_OFFSET;
    nv_wind_brake_val &= ~NV_SOFT_NUM_STEP_MASK;
    nv_wind_brake_val |= num_steps << NV_SOFT_NUM_STEP_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_POSITION >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_positin_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_positin_val >> 0)& 0xff;      
    ack += i2c_word_write(i2cMessage);
    *(i2cMessage+0) = (char)(I2C_START_UP >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_start_up_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_start_up_val >> 0)& 0xff;          
    ack += i2c_word_write(i2cMessage);
    *(i2cMessage+0) = (char)(I2C_WIND_BRAKE >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_wind_brake_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_wind_brake_val >> 0)& 0xff;          
    ack += i2c_word_write(i2cMessage);
    
    return ack; 
}

/** i2c to set the High Torque Start Up (pulse train)
*/
int i2c_mbed_fpga::i2c_set_high_torque_start_up(unsigned int enbale, unsigned int mantisaa_3b, unsigned int exponent_3b){
    int ack = 0;
    nv_start_up_val &= ~NV_LONG_START_MASK;
    nv_start_up_val |= enbale << NV_LONG_START_OFFSET;
    nv_positin_val &= ~NV_FIRST_NON_FLAT_TIME_MASK;
    nv_positin_val |= ((exponent_3b << 3) | mantisaa_3b) << NV_FIRST_NON_FLAT_TIME_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_POSITION >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_positin_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_positin_val >> 0)& 0xff;  
    ack += i2c_word_write(i2cMessage);
    *(i2cMessage+0) = (char)(I2C_START_UP >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_start_up_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_start_up_val >> 0)& 0xff;             
    ack += i2c_word_write(i2cMessage);
        
    return ack; 
}

/** i2c to set the Single Pulse Start Up (pulse train)
*/
int i2c_mbed_fpga::i2c_set_single_pulse_start_up(unsigned int enbale, unsigned int mantisaa_3b, unsigned int exponent_3b){
    int ack = 0;
    nv_start_up_val &= ~NV_SINGLE_PULSE_START_MASK;
    nv_start_up_val |= enbale << NV_SINGLE_PULSE_START_OFFSET;
    nv_positin_val &= ~NV_FIRST_NON_FLAT_TIME_MASK;
    nv_positin_val |= ((exponent_3b << 3) | mantisaa_3b) << NV_FIRST_NON_FLAT_TIME_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_POSITION >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_positin_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_positin_val >> 0)& 0xff;  
    ack += i2c_word_write(i2cMessage);
    *(i2cMessage+0) = (char)(I2C_START_UP >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_start_up_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_start_up_val >> 0)& 0xff;        
    ack += i2c_word_write(i2cMessage);
        
    return ack; 
}

/** i2c to set the rough regulation gain
*/
int i2c_mbed_fpga::i2c_set_rough_gain(unsigned int rough_gain){
    int ack = 0;
    nv_gen_ctrl_val &= ~NV_ROUGH_GAIN_MASK;
    nv_gen_ctrl_val |= rough_gain << NV_ROUGH_GAIN_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0)= (char)(I2C_GEN_CTRL >> 0)& 0xff;
    *(i2cMessage+1)= (char)(nv_gen_ctrl_val >> 8)& 0xff;
    *(i2cMessage+2)= (char)(nv_gen_ctrl_val >> 0)& 0xff;  
    ack += i2c_word_write(i2cMessage);
    
    return ack; 
}

/** i2c to set the ehp regulation gain
*/
int i2c_mbed_fpga::i2c_set_ehp_reg_gain(unsigned int ehp_gain){
    int ack = 0;
    nv_gen_ctrl_val &= ~NV_EHP_REG_GAIN_MASK;
    nv_gen_ctrl_val |= ehp_gain << NV_EHP_REG_GAIN_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0)= (char)(I2C_GEN_CTRL >> 0)& 0xff;
    *(i2cMessage+1)= (char)(nv_gen_ctrl_val >> 8)& 0xff;
    *(i2cMessage+2)= (char)(nv_gen_ctrl_val >> 0)& 0xff;  
    ack += i2c_word_write(i2cMessage);
    
    return ack; 
}

/** i2c to set the ehp regulation gain
*/
int i2c_mbed_fpga::i2c_set_fall_time_blank(unsigned int blank_time){
    int ack = 0;
    nv_dig_config_val &= ~NV_FLAT_BLANK_MASK;
    nv_dig_config_val |= blank_time << NV_FLAT_BLANK_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0)= (char)(I2C_DIGITAL_CFG >> 0)& 0xff;
    *(i2cMessage+1)= (char)(nv_dig_config_val >> 8)& 0xff;
    *(i2cMessage+2)= (char)(nv_dig_config_val >> 0)& 0xff;  
    ack += i2c_word_write(i2cMessage);
    
    return ack; 
}


/** i2c to set the current threshold for I_didt
*/
int i2c_mbed_fpga::i2c_set_comm_i_thres(unsigned int i_thr_low, unsigned int i_thr_high){
    int ack = 0;
    nv_i_zc_th_low_val &= ~NV_I_ZC_TH_LOW_MASK;
    nv_i_zc_th_low_val |= i_thr_low << NV_I_ZC_TH_LOW_OFFSET;
    nv_i_zc_th_high_val &= ~NV_I_ZC_TH_HIGH_MASK;
    nv_i_zc_th_high_val |= i_thr_high << NV_I_ZC_TH_HIGH_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_I_ZC_TH_LOW >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_i_zc_th_low_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_i_zc_th_low_val >> 0)& 0xff;  
    ack += i2c_word_write(i2cMessage);
    *(i2cMessage+0) = (char)(I2C_I_ZC_TH_HIGH >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_i_zc_th_high_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_i_zc_th_high_val >> 0)& 0xff;  
    ack += i2c_word_write(i2cMessage);
    return ack; 
}

/** i2c to set the di current threshold for didt
*/
int i2c_mbed_fpga::i2c_set_comm_di_thres(unsigned int di_1st, unsigned int di_2nd){
    int ack = 0;
    nv_di_th_1st_val &= ~NV_DI_TH_1ST_MASK;
    nv_di_th_1st_val |= di_1st << NV_DI_TH_1ST_OFFSET;
    nv_di_th_2nd_val &= ~NV_DI_TH_2ND_MASK;
    nv_di_th_2nd_val |= di_2nd << NV_DI_TH_2ND_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_DI_TH_1ST >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_di_th_1st_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_di_th_1st_val >> 0)& 0xff;   
    ack += i2c_word_write(i2cMessage);
    *(i2cMessage+0) = (char)(I2C_DI_TH_2ND >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_di_th_2nd_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_di_th_2nd_val >> 0)& 0xff;   
    ack += i2c_word_write(i2cMessage);
    return ack; 
}




/** i2c to clean the I2C controller settins
*/
int i2c_mbed_fpga::i2c_clear_spd_ctrl(){
    int ack = 0;
    nv_spd_control_1_val = 0;
    nv_spd_control_2_val = 0;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_SPD_CTRL_1 >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_spd_control_1_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_spd_control_1_val >> 0)& 0xff;  
    ack += i2c_word_write(i2cMessage);
    *(i2cMessage+0) = (char)(I2C_SPD_CTRL_2 >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_spd_control_2_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_spd_control_2_val >> 0)& 0xff;  
    ack += i2c_word_write(i2cMessage);
    
    return ack; 
}

/** i2c to set the I2C speed input mode
*/
int i2c_mbed_fpga::i2c_set_input_mode(unsigned int mode){
    int ack = 0;
    nv_application_cfg_val &= ~NV_INPUT_MODE_CFG_MASK;
    nv_application_cfg_val |= mode << NV_INPUT_MODE_CFG_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_APPLICATION_CFG >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_application_cfg_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_application_cfg_val >> 0)& 0xff;  
    ack += i2c_word_write(i2cMessage);    
    return ack; 
}



/** i2c to set the open loop mode
*/
int i2c_mbed_fpga::i2c_set_loop_mode(unsigned int openloop){
    int ack = 0;
    nv_spd_control_1_val &= ~NV_SPD_LOOP_MODE_MASK;
    nv_spd_control_1_val |= openloop << NV_SPD_LOOP_MODE_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_SPD_CTRL_1 >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_spd_control_1_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_spd_control_1_val >> 0)& 0xff;  
    ack += i2c_word_write(i2cMessage);
    
    return ack; 
}

/** i2c to set the Single Pulse Start Up (pulse train)
*/
int i2c_mbed_fpga::i2c_set_open_loop_duty(unsigned int duty){
    int ack = 0;
    ram_open_duty_val = duty;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_SPEED_DUTY)& 0xff;
    *(i2cMessage+1) = (char)(ram_open_duty_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(ram_open_duty_val >> 0)& 0xff;  
    ack += i2c_word_write(i2cMessage);
        
    return ack; 
}

/** i2c to set the speed curve type
*/
int i2c_mbed_fpga::i2c_set_curve_type(unsigned int curvetype){
    int ack = 0;
    nv_spd_control_1_val &= ~NV_CURVE_MODE_MASK;
    nv_spd_control_1_val |= curvetype << NV_CURVE_MODE_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_SPD_CTRL_1 >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_spd_control_1_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_spd_control_1_val >> 0)& 0xff;   
    ack += i2c_word_write(i2cMessage);
    
    return ack; 
}

/** i2c to set the open dc ini
*/
int i2c_mbed_fpga::i2c_set_dc_ini(unsigned int ini){
    int ack = 0;
    nv_spd_control_2_val &= ~NV_DC_OPENLOOP_INI_MASK;
    nv_spd_control_2_val |= ini << NV_DC_OPENLOOP_INI_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_SPD_CTRL_2 >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_spd_control_2_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_spd_control_2_val >> 0)& 0xff;  
    ack += i2c_word_write(i2cMessage);
    
    return ack; 
}

/** i2c to set the open dc slew rate
*/
int i2c_mbed_fpga::i2c_set_dc_sr(unsigned int sr){
    int ack = 0;
    nv_spd_control_2_val &= ~NV_DC_OPENLOOP_SR_MASK;
    nv_spd_control_2_val |= sr << NV_DC_OPENLOOP_SR_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_SPD_CTRL_2 >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_spd_control_2_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_spd_control_2_val >> 0)& 0xff; 
    ack += i2c_word_write(i2cMessage);
    
    return ack; 
}

/** i2c to set the target CLIM during start up
*/
int i2c_mbed_fpga::i2c_set_clim_start_up(unsigned int clim){
    int ack = 0;
    nv_clim_user_1_val &= ~NV_TARGET_CLIM_USER_PULSES_MASK;
    nv_clim_user_1_val |= clim << NV_TARGET_CLIM_USER_PULSES_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_CLIM_USER_1 >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_clim_user_1_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_clim_user_1_val >> 0)& 0xff; 
    ack += i2c_word_write(i2cMessage);
    
    return ack; 
}

/** i2c to set the target CLIM during brake
*/
int i2c_mbed_fpga::i2c_set_clim_brake(unsigned int clim){
    int ack = 0;
    nv_clim_user_0_val &= ~NV_TARGET_CLIM_USER_BRAKE_MASK;
    nv_clim_user_0_val |= clim << NV_TARGET_CLIM_USER_BRAKE_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_CLIM_USER_0 >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_clim_user_0_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_clim_user_0_val >> 0)& 0xff; 
    ack += i2c_word_write(i2cMessage);
    
    return ack; 
}

/** i2c to set the target CLIM during run time
*/
int i2c_mbed_fpga::i2c_set_clim_run_time(unsigned int clim){
    int ack = 0;
    nv_clim_user_0_val &= ~NV_TARGET_CLIM_USER_MOTOR_MASK;
    nv_clim_user_0_val |= clim << NV_TARGET_CLIM_USER_MOTOR_OFFSET;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(I2C_CLIM_USER_0 >> 0)& 0xff;
    *(i2cMessage+1) = (char)(nv_clim_user_0_val >> 8)& 0xff;
    *(i2cMessage+2) = (char)(nv_clim_user_0_val >> 0)& 0xff; 
    ack += i2c_word_write(i2cMessage);
    
    return ack; 
}


/** i2c to read I2C_CLIM_CALIB
*/

