#include "mbed.h"
#include "SDFileSystem.h"
#include "BNO055.h"
#include "DS1307.h"
#include "string.h"

/*SD card*/
#define DI D11
#define DO D12
#define SCK D13
#define CS D10

/*RTC*/
#define I2C1_SDA PB_9
#define I2C1_SCL PB_8

RtcDs1307 *rtc;

char fname[255];
char current_time[64];

/*IMU*/
#define I2C2_SDA D5
#define I2C2_SCL D7

/*Serial buffer*/
#define PAGE_SIZE 4096
#define HEADER_SIZE 10
#define CHUNK_SIZE (PAGE_SIZE-HEADER_SIZE)
#define BUF_SIZE (16*PAGE_SIZE)

int start = 0, end = 0;
unsigned char buf[BUF_SIZE];
unsigned char hdr[HEADER_SIZE] = {0xff, 0xff,'M','A','N','B','L','O','R','T'};

/*UART*/
#define BAUD_RATE 921600
Serial serial4(D8, D2);

/*Test pins*/
DigitalOut test(PC_8);
DigitalOut test2(PC_6);
int flag=0, flag2=0;

void printDT(char *pre, DateTime &dt) {
    printf("%s %u/%u/%02u %2u:%02u:%02u\r\n"
        ,pre
        ,dt.month(),dt.day(),dt.year()
        ,dt.hour(),dt.minute(),dt.second()
        );
}

bool rtcUpdate(RtcDs1307 &rtc, int32_t bias) {   
    bool bUpdated = false;
    int64_t compiledTime = DateTime(__DATE__,__TIME__).unixtime();
 
    time_t localt = DateTime(compiledTime + bias).unixtime();
    
    if(*((time_t *)&rtc[0]) != localt) {
        rtc.adjust(localt);
        *((time_t *)&rtc[0]) = localt;
        bUpdated = rtc.commit();
    }
    
    return bUpdated;
}

void rx_callback() {
    while(serial4.readable()) {
        buf[end] = serial4.getc();
        end++;
        if (end == BUF_SIZE) end = 0;
    }
}

int main() {
    /*init SD card, wait for card insertion*/    
    SDFileSystem sd(DI, DO, SCK, CS, "sd");
    while (sd.disk_status()) {
        sd.disk_initialize();
        wait(0.5);
    }
    
    /*init RTC, get time and date for filename*/
    I2C i2c1(I2C1_SDA, I2C1_SCL);
    i2c1.frequency(100000);
    rtc = new RtcDs1307(i2c1);
    
    rtcUpdate(*rtc, -(5*60*60));
    DateTime dt = rtc->now();
    printDT("It is now", dt);
    
    strcpy(fname, "/sd/log_");
    sprintf(current_time, "%u.%u.%02u_%02u.%02u.%02u"
        ,dt.month(),dt.day(),dt.year()
        ,dt.hour(),dt.minute(),dt.second()
    );    
    strcat(fname, current_time);
    strcat(fname, ".dat");
    printf("Logging to %s\n", fname);
    
    FILE *fp = fopen(fname, "wb");
    
    /*init IMU*/
    BNO055 imu(I2C2_SDA, I2C2_SCL);
    imu.reset();
    if (!imu.check()) {
        printf("%s\n", "IMU not ready");  
    } else {
        printf("BNO055 found\r\n\r\n");
        printf("Chip          ID: %03d\r\n",imu.ID.id);
        printf("Accelerometer ID: %03d\r\n",imu.ID.accel);
        printf("Gyroscope     ID: %03d\r\n",imu.ID.gyro);
        printf("Magnetometer  ID: %03d\r\n\r\n",imu.ID.mag);
        printf("Firmware version v%d.%0d\r\n",imu.ID.sw[0],imu.ID.sw[1]);
        printf("Bootloader version v%d\r\n\r\n",imu.ID.bootload);
    }
    
    imu.setmode(OPERATION_MODE_NDOF);
    imu.set_accel_units(MPERSPERS);
    
    /*init serial*/    
    serial4.baud(BAUD_RATE);
    serial4.attach(rx_callback);
     
    for (;;) {
        int length = end - start;
        if (length < 0) length += BUF_SIZE;
        if (length >= CHUNK_SIZE) {
            fwrite(hdr, 1, HEADER_SIZE, fp);
            if (start+CHUNK_SIZE-1 < BUF_SIZE) {
                fwrite(buf+start, 1, CHUNK_SIZE, fp);
            } else {
                fwrite(buf+start, 1, BUF_SIZE-start+1, fp);
                fwrite(buf, 1, CHUNK_SIZE-(BUF_SIZE-start+1), fp);
            }
            fflush(fp);
            start += CHUNK_SIZE;
            if (start >= BUF_SIZE) start -= BUF_SIZE;
        }
    }
}
