#include "mbed.h"
#include "SDFileSystem.h"
#include "DS1307.h"
#include "BNO055.h"
#include "string.h"

#define M_PI 3.1415927f

#define DI D11
#define DO D12
#define SCK D13
#define CS D10

#define I2C1_SDA PB_9
#define I2C1_SCL PB_8

#define I2C2_SDA D5
#define I2C2_SCL D7

#define PAGE_SIZE 4096
#define clip(a) if ((a > 254)) a = 254

PwmOut pwm_r(D4);
PwmOut pwm_g(D6);
PwmOut pwm_b(D3);

char fname[255];
char current_time[64];

unsigned char buf[PAGE_SIZE];
int buf_idx = 0;

void printDT(char *pre, DateTime &dt)
{
    printf("%s %u/%u/%02u %2u:%02u:%02u\r\n"
        ,pre
        ,dt.month(),dt.day(),dt.year()
        ,dt.hour(),dt.minute(),dt.second()
        );
}

bool rtcUpdate(RtcDs1307 &rtc, int32_t bias) // this must be signed
{   bool bUpdated = false;
 
    // Use the compiled date/time as a basis for setting the clock.
    // We assign it to a signed integer so that negative biases work correctly
    int64_t compiledTime = DateTime(__DATE__,__TIME__).unixtime();
 
    // This assumes that the program is run VERY soon after the initial compile.
    time_t localt = DateTime(compiledTime + bias).unixtime(); // offset by bias
    
    // If the stored static time stamp does not equal the compiled time stamp,
    // then we need to update the RTC clock and the stored time stamp
    if(*((time_t *)&rtc[0]) != localt)
    {
        // Update the RTC time as local time, not GMT/UTC
        rtc.adjust(localt);
        // Store the new compiled time statically in the object ram image
        *((time_t *)&rtc[0]) = localt;
        // Push the object ram image to the RTC ram image
        bUpdated = rtc.commit();
    }
    
    return bUpdated;
}

int main() {
    printf("%s\n", "Manworm's High-Speed Logger");
    
    /*init RGB lights*/
    pwm_r.period(0.01);
    pwm_g.period(0.01);
    pwm_b.period(0.01);
    
    pwm_r = 0.0f;
    pwm_g = 0.0f;
    pwm_b = 0.0f;
    
    /*wait for SD card to be inserted*/
    SDFileSystem sd(DI, DO, SCK, CS, "sd");
    while (sd.disk_status()) {
        sd.disk_initialize();
        wait(0.5);
    }
    
    /*init RTC, get time and date for filename*/
    I2C i2c1(I2C1_SDA, I2C1_SCL);
    i2c1.frequency(100000);
    RtcDs1307 rtc(i2c1);
    
    rtcUpdate(rtc, -(5*60*60));
    DateTime dt = rtc.now();
    printDT("It is now", dt);
    
    strcpy(fname, "/sd/log_");
    sprintf(current_time, "%u.%u.%02u_%02u.%02u.%02u"
        ,dt.month(),dt.day(),dt.year()
        ,dt.hour(),dt.minute(),dt.second()
    );    
    strcat(fname, current_time);
    strcat(fname, ".csv");
    printf("Logging to %s\n", fname);
    
    FILE *fp = fopen(fname, "w");
    
    /*init IMU*/
    BNO055 imu(I2C2_SDA, I2C2_SCL);
    imu.reset();
    if (!imu.check()) {
        printf("%s\n", "IMU not ready");  
    } else {
        printf("BNO055 found\r\n\r\n");
        printf("Chip          ID: %03d\r\n",imu.ID.id);
        printf("Accelerometer ID: %03d\r\n",imu.ID.accel);
        printf("Gyroscope     ID: %03d\r\n",imu.ID.gyro);
        printf("Magnetometer  ID: %03d\r\n\r\n",imu.ID.mag);
        printf("Firmware version v%d.%0d\r\n",imu.ID.sw[0],imu.ID.sw[1]);
        printf("Bootloader version v%d\r\n\r\n",imu.ID.bootload);
    }
    imu.setmode(OPERATION_MODE_NDOF);
    imu.set_accel_units(MPERSPERS);
    
    float theta = 0.0f;
    int tics = 0;
    for (;;) {
        float r, g, b;
        r = sinf(theta);
        g = sinf(theta + 2 * M_PI / 3);
        b = sinf(theta - 2 * M_PI / 3);
        
        pwm_r = 0.5f + 0.5f * r;
        pwm_g = 0.5f + 0.5f * g;
        pwm_b = 0.5f + 0.5f * b;
        
        theta += 0.001f;
        if (theta > 2 * M_PI) theta -= 2 * M_PI;
        
        tics++;
        if (tics % 100 == 0) {
            tics = 0;
            
            imu.get_accel();
            fprintf(fp, "%d,%d,%d\n",(int) (100*imu.accel.x),(int)(100*imu.accel.y),(int)(100*imu.accel.z));
            fflush(fp);
        }
        wait(0.001);
    }
}