/*******************************************************************************
 * Mangages LCD display, joystick and accelerometer for user interface.
 * Bryan Wade
 * 27 MAR 2014
 ******************************************************************************/
#include "mbed.h"
#include "C12832_lcd.h"
#include "MMA7660.h"
#include "user_interface.h"
#include "effects.h"

C12832_LCD lcd;
MMA7660 accel(p28, p27, true);
BusIn joystick(p15, p12, p13, p16);
AnalogIn pot(p19);

static const int STICK_IDLE  = 0;
static const int STICK_UP    = 1;
static const int STICK_DOWN  = 2;
static const int STICK_LEFT  = 4;
static const int STICK_RIGHT = 8;

static uint8_t stickPrevious;
static effect_mode_t selectedMode;
static char *selectedModeName;
static uint16_t effectGain;

/*
 * Update the display with mode, gain and buffer level.
 */
void updateDisplay(int32_t level)
{
    lcd.locate(0,0);
    lcd.printf("Effect: %s      ", selectedModeName);
       
    lcd.locate(0,10);
    lcd.printf("Intensity: %d    ", effectGain / (MAX_EFFECT_GAIN / 100));
    
    lcd.locate(0,20);
    lcd.printf("Buffer: %d    ", level);
}

/*
 * Get effects mode selected by joystick.
 */
void serviceJoystick(void)
{
    uint8_t stick = joystick;
    
    if (stick != stickPrevious) {
        stickPrevious = stick;    
    
        switch (stick)
        {
        case STICK_UP:
            selectedMode = EFFECT_ECHO;
            selectedModeName = "Echo";
            break;
            
        case STICK_DOWN:
            selectedMode = EFFECT_REVERB;
            selectedModeName = "Reverb";
            break;
            
        case STICK_LEFT:
        case STICK_RIGHT:
            selectedMode = EFFECT_STRAIGHT;
            selectedModeName = "Straight";
            break;
        }    
    }
}

/*
 * Get effects gain from accelerometer tilt.
 */
 void serviceAccel(void)
 {
     // Tilt sensitivity selected to produce max effect gain 
     // at about 80 deg tilt.
     static const int32_t TILT_SENSITIVITY = 2750; 
     int a[3];
     accel.readData(a);
     //Convert x-axis raw accelerometer data to the effect gain.
     int32_t x = a[0] * TILT_SENSITIVITY;
     if (x < 0) x = -x;
     if (x > MAX_EFFECT_GAIN) x = MAX_EFFECT_GAIN;
     effectGain = (uint16_t)x;    
 }
 
/*
 * Initialize the joystick and display.
 */
void UI_Initialize(void)
{
    lcd.cls();
    stickPrevious = STICK_IDLE;
    selectedMode = EFFECT_STRAIGHT;
    selectedModeName = "Straight";
    effectGain = 0;
    // Set the accelerometer to sample slightly
    // faster than we poll the user interface.
    accel.setSampleRate(32);
}

/*
 * Updates display, joystick and accelerometer
 */
void UI_Update(int32_t bufferLevel)
{
    serviceJoystick();
    serviceAccel();
    updateDisplay(bufferLevel);    
}
 

effect_mode_t UI_GetEffectMode(void) { return selectedMode; }

uint16_t UI_GetEffectGain(void) { return effectGain; }


    
