/*******************************************************************************
 * Project: USB Sound Effects.
 *
 * Uses the mbed LPC1768 and mbed-application-board to create a USB audio device
 * that streams audio from a host computer to headphones or powered speakers. 
 * A couple different sound effects can be applied to the stream in real-time,
 * tilting the mbed alters the effects.
 *
 *                                               ECHO
 *       The joystick selects )                   |
 *       one of three effect  )       STRAIGHT -  o  - STRAIGHT
 *       modes.               )                   |
 *                                              REVERB   
 * 
 *
 *
 *                                               \\           ||    
 *       Tilting the mbed     )      ======       \\          ||
 *       determines intensity )                    \\         ||
 *       of the effect.       )
 *                                     0%         50%         100%  
 *
 * The LCD display shows the current effect mode, intesity and buffer level.
 *
 * For better audio quality, packets are buffered before playbeck. Playback is
 * halted when the buffer drops below two packets and resumes when it is greater
 * than four packets. This creates a lag of a few milliseconds, which is not 
 * detectable by a listener using straight playback mode.
 * 
 * The echo effect produces echos with a delay of about 250 milliseconds. Tilting
 * the mbed alters the persistence of echo.
 *
 * The reverb effect utilizes multiple delay lines of length varying from  
 * about 25 to 200 milliseconds. Again, tilting the mbed alters the 
 * persistence. 
 *
 * Implemenation Notes:
 * Audio playback is implement with an mbed ticker, and it is critical that this interrupt
 * is not delayed to acheive decent audio quality. USB audio packet handling also uses
 * interrupts, but the priority is set lower to ensure proper playback. The USBAudio class
 * was modified to use a callback to transfer the received packet from the packet buffer to 
 * the larger playback buffer. The user interface is handled entirely in the user-context
 * since it has only soft timing requirements.
 *
 * Future work:
 * 1. Clocks within the host computer and mbed will differ slightly, so the 
 * average sample rate from the computer will not exactly match the average mbed
 * playback rate. Over time the playback buffer level with creep until it becomes 
 * totaly empty or full. The current implementation addresses this by simply 
 * increasing the buffer size and selecting a sample rate that the mbed 
 * can replicate most accurately (mulitple of 1 us to use mbed Ticker). A better 
 * solution may be to slightly alter the playback rate in response to the buffer level.
 * For example use 20us rate when the buffer level is high and 19us when the buffer 
 * level is low. Of course this would distort the playback frequecy by 5%, so it would
 * be even better to make a custom timer interrupt with better resolution than the 
 * mbed ticker.
 *
 * 2. It would be interesting to add stereo audio, but the mbed-application-board 
 * audio jack is mono so this would require hardware customization.
 *
 * Bryan Wade
 * 27 MAR 2014
 ******************************************************************************/

#include "mbed.h"
#include "audio.h"
#include "effects.h"
#include "user_interface.h"

int main() 
{
    UI_Initialize();    // Init the user interface.
    Audio_Initialize(); // Starts audio playback.
    
    while (true) 
    {   
        // Check the playback buffer level.      
        int32_t level = Audio_CheckPlaybackBufferLevel();
        
        // Update the user interface, passing in the current buffer level 
        // for display
        UI_Update(level);          

        // Set the effect mode and gain according to the user input.
        Effects_SetMode(UI_GetEffectMode());
        Effects_SetGain(UI_GetEffectGain());

        // 50 milliseconds provides good UI responsivness.
        wait(.05);
    }
}

