/*******************************************************************************
 * Implements delay lines for audio samples.
 * Bryan Wade
 * 27 MAR 2014
 ******************************************************************************/
#include "delay.h"
#include "string.h"

struct delay_t { // Implementation of opaque type.
    int16_t *base;
    int16_t *head;
    size_t len;
};

/*
 * Create a delay object given a ptr and size of available RAM.
 * This is only allocating memory for the metadata. The location of 
 * the buffer memory is passed in.
 * @return Ptr to the new delay.
 */
delay_t *Delay_Create(void)
{
    return (delay_t *)malloc(sizeof(delay_t));
}

/*
 * Configure delay buffer location and size.
 * @return True if successful.
 */
bool Delay_Configure(delay_t *delay, void *ram, size_t size)
{
    if (delay) {
        delay->base = delay->head = (int16_t *)ram;
        delay->len = size / sizeof(int16_t);
        memset(ram, 0, size);
        return true;
    }
    
    return false;
}

/*
 * Push a new sample with feedback into the delay while simulateously retieving the output sample.
 * @return Output data.
 */
int16_t Delay_WriteWithFeedback(delay_t *delay, int16_t dataIn, uint16_t gain)
{
    int16_t dataOut = *delay->head;

    // Feedback gain is fixed-point Q16 format.
    *delay->head = dataIn + ((gain * dataOut ) >> 16);
    
    if (++delay->head >= delay->base + delay->len)
        delay->head = delay->base;
        
    return dataOut;
}

