#include "MicroBit.h"
#include "MicroBitDisplay.h"

#ifndef microbitOLED_Display
#define microbitOLED_Display

/**
  * Constructs display by parameters to the '5 rows'.
  * Takes five 5bit values in range [0,31] and a bool \n 
  * value for 'showing left bit column' 
  */
struct Ledrows {
    union {
    struct {
    int r0:5;
    int r1:5;
    int r2:5;
    int r3:5;
    int r4:5; };
    int all; };
    Ledrows(uint8_t lr0,uint8_t lr1,uint8_t lr2,uint8_t lr3,uint8_t lr4, bool b15On) 
    : r0(lr0),r1(lr1),r2(lr2),r3(lr3),r4(lr4) {
        if (b15On) all |= 0x01084210; }
    
};


/**
  * Use Microbit 5x5 led display in various output communications.
  * It deals with display in DISPLAY_MODE_BLACK_AND_WHITE mode 
  */
class Display { 

    /**
      * Representation the 25 leds.
      * The physical leds are counted from right bottom and continuing \n 
      * from right on each row above. Light in a led corresponds to 1 stick.\n 
      * Writting to the display is writting to this buffer AND the display  \n 
      * only if a value is changed - this demands clearing the display is \n 
      * nullifying this stick prior uDisplay.image.clear();
      */
    bool stick[25];
    MicroBitDisplay & uDisplay;
    union { // context dependt trueshowing aliases
    uint8_t curLength;
    uint8_t rowNr; };
    uint32_t showbit;
    
public:

    /**
      * Display hold a refference to a MicroBitDisplay object. 
      * When constructed from MicroBit's  MicroBitDisplay of a not yet initialized \n 
      * MicroBit object - that must be done before using the Display instance.\n 
      *
      * @param uDpl reference to a MicroBitDisplay object.
      */
    Display(MicroBitDisplay & uDpl);
    
    /**
      * Clears the display and its buffer 
      */
    void clear();
    
    /**
      * The stick's length is represented by the count of digits being 1.
      *
      * @param size in range [0,24]
      */
    void setStick(uint8_t size);
    
    /**
      * Set a single dot.
      *
      * @param position in [0,24] for showing that single dot.
      */
    void setFlag(uint8_t position);
    
    /**
      * Puts a bitpattern in a row.
      *
      * @param rowNr in [0,4] for selected row
      * @param rowContent in [0,31] to be shown binary
      */
    void toRow(uint8_t rowNr, uint8_t rowContent);
    
    /**
      * 25 bits showed in the display as a whole.
      *
      * @param bits25 contains the 25 bit to be showed
      */
    void toRows(uint32_t allBits);

    /**
      * Vertdecimal shows a single number in [00000,99999].
      * The parameters goes from top to bottom, whereas a obvious \n  
      * perception is reading the disply number from top bottom. \n   
      * Nothing restrict the rows to hold values less teen \n 
      * but if some are greather than 31 the value or proberbly \n
      * bitwise ored og cleared by other numbers
      *
      * @param r0 bottom row - row 0 
      * @param r1 row 1
      * @param r2 row 2
      * @param r3 row 3 
      * @param r4 top row - row 4
      */
    void vertDecimal(uint8_t r0,uint8_t r1,uint8_t r2,uint8_t r3,uint8_t r4); 
  
    /**
      * Vertdecimal shows a single number in [00000,99999].
      * The parameters goes from top to bottom, whereas a obvious \n  
      * perception is reading the disply number from top bottom. \n   
      * It os possible to turn on left column on the display
      * 
      * @param r0 bottom row - row 0 
      * @param r1 row 1
      * @param r2 row 2
      * @param r3 row 3 
      * @param r4 top row - row 4
      * @param leftDotOn turns left dot on. leftdot is not used to represent a digit in base 10 numbers.
      */
    void vertDecimal(uint8_t r0,uint8_t r1,uint8_t r2,uint8_t r3,uint8_t r4, bool leftDotOn);

    /**
    * Binary Clock with hours and minuts.
    * hours in top rows (4 for multiplum of 10, 3 for remainer) \n 
    * minuts in bottom rows (1 for multiplum of 10, 0 for remainer)
    *
    * @param minuts of the day
    */
    void vertClock(uint16_t minuts); 
    
    /**
    * Binary Clock with hours, minuts an seconds.
    * hours in row 4 (top row)\n 
    * minuts in rows 3 and 2  (3 for multiplum of 10, 2 for remainer)\n 
    * seconds in rows 1 and 0 (1 for multiplum of 10, 0 for remainer)\n 
    *
    * @param second of the day
    */
    void vertSecClock(uint32_t minuts); 
    
private:
    void paintStick(bool(Display::*getState)(uint8_t));
    void clearStick();
    bool bitplot(uint8_t num);
    bool simplePlot(uint8_t num);
    bool lessCurLength(uint8_t num);
    bool orPoint(uint8_t num);
};    


#endif
