/**
  ******************************************************************************
  * @file    ov7670.c
  * @author  MCD Application Team
  * @version V1.0.0
  * @date    25-June-2015
  * @brief   This file provides the OV7670 camera driver
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */ 

/* Includes ------------------------------------------------------------------*/
#include "ov7670.h"

/** @addtogroup BSP
  * @{
  */ 

/** @addtogroup Components
  * @{
  */ 
  
/** @addtogroup OV7670
  * @brief     This file provides a set of functions needed to drive the 
  *            OV7670 Camera module.
  * @{
  */

/** @defgroup OV7670_Private_TypesDefinitions
  * @{
  */ 

/**
  * @}
  */ 

/** @defgroup OV7670_Private_Defines
  * @{
  */

/**
  * @}
  */ 
  
/** @defgroup OV7670_Private_Macros
  * @{
  */
     
/**
  * @}
  */  
  
/** @defgroup OV7670_Private_FunctionPrototypes
  * @{
  */
static uint64_t ov7670_ConvertValue(uint32_t feature, uint32_t value);
/**
  * @}
  */ 
  
/** @defgroup OV7670_Private_Variables
  * @{
  */        

CAMERA_DrvTypeDef   ov7670_drv = 
{
  ov7670_Init,
  ov7670_ReadID,  
  ov7670_Config,
};

/* Initialization sequence for VGA resolution (640x480)*/
const unsigned char OV7670_VGA[][2]=
{
  {0x12, 0x04},       //QVGA size, RGB mode
  {0x40, 0xd0},       //RGB565
  {0xb0, 0x84},       //Color mode
  {0x11, 0x00},       //PCLK settings, 30fps
  {0x32, 0x80},       //HREF
  {0x17, 0x17},       //HSTART
  {0x18, 0x05},       //HSTOP
  {0x03, 0x0a},       //VREF
  {0x19, 0x02},       //VSTART
  {0x1a, 0x7a},       //VSTOP
  // Additional parameters
  {0x34, 0x11},       {0x3f, 0x00},
  {0x75, 0x05},       {0x76, 0xe1},
  {0x4c, 0x00},       {0x77, 0x01},
  {0xb8, 0x0a},       {0x41, 0x18},
  {0x3b, 0x12},       {0xa4, 0x88},
  {0x96, 0x00},       {0x97, 0x30},
  {0x98, 0x20},       {0x99, 0x30},
  {0x9a, 0x84},       {0x9b, 0x29},
  {0x9c, 0x03},       {0x9d, 0x4c},
  {0x9e, 0x3f},       {0x78, 0x04},
  {0x0e, 0x61},       {0x0f, 0x4b},
  {0x16, 0x02},       {0x1e, 0x00},
  {0x21, 0x02},       {0x22, 0x91},
  {0x29, 0x07},       {0x33, 0x0b},
  {0x35, 0x0b},       {0x37, 0x1d},
  {0x38, 0x71},       {0x39, 0x2a},
  {0x3c, 0x78},       {0x4d, 0x40},
  {0x4e, 0x20},       {0x69, 0x00},
  {0x6b, 0x3a},       {0x74, 0x10},
  {0x8d, 0x4f},       {0x8e, 0x00},
  {0x8f, 0x00},       {0x90, 0x00},
  {0x91, 0x00},       {0x96, 0x00},
  {0x9a, 0x00},       {0xb1, 0x0c},
  {0xb2, 0x0e},       {0xb3, 0x82},

  {0x4f, 0x99},
  {0x50, 0x99},
  {0x51, 0x00},
  {0x52, 0x28},
  {0x53, 0x71},
  {0x54, 0x99},
  {0x58, 0x9e},
  {0x56,0x40},
  {0x58,0x9e},
  {0x59,0x88},
  {0x5a,0x88},
  {0x5b,0x44},
  {0x5c,0x67},
  {0x5d,0x49},
  {0x5e,0x0e},
  {0x69,0x00},
  {0x6a,0x40},
  {0x6b,0x0a},
  {0x6c,0x0a},
  {0x6d,0x55},
  {0x6e,0x11},
  {0x6f,0x9f},


};

/* Initialization sequence for QVGA resolution (320x240) */
const unsigned char OV7670_QVGA[][2]=
{
  
// Image format
{0x12, 0x14},       //QVGA size, RGB mode
{0x40, 0xd0},       //RGB565
{0xb0, 0x84},       //Color mode

// Hardware window
{0x11, 0x01},       //PCLK settings, 15fps
{0x32, 0x80},       //HREF
{0x17, 0x17},       //HSTART
{0x18, 0x05},       //HSTOP
{0x03, 0x0a},       //VREF
{0x19, 0x02},       //VSTART
{0x1a, 0x7a},       //VSTOP

// Scalling numbers
{0x70, 0x3a},       //X_SCALING
{0x71, 0x35},       //Y_SCALING
{0x72, 0x11},       //DCW_SCALING
{0x73, 0xf0},       //PCLK_DIV_SCALING
{0xa2, 0x02},       //PCLK_DELAY_SCALING

// Matrix coefficients
{0x4f, 0x80},       {0x50, 0x80},
{0x51, 0x00},       {0x52, 0x22},
{0x53, 0x5e},       {0x54, 0x80},
{0x58, 0x9e},

// Gamma curve values
{0x7a, 0x20},       {0x7b, 0x10},
{0x7c, 0x1e},       {0x7d, 0x35},
{0x7e, 0x5a},       {0x7f, 0x69},
{0x80, 0x76},       {0x81, 0x80},
{0x82, 0x88},       {0x83, 0x8f},
{0x84, 0x96},       {0x85, 0xa3},
{0x86, 0xaf},       {0x87, 0xc4},
{0x88, 0xd7},       {0x89, 0xe8},

// AGC and AEC parameters
{0xa5, 0x05},       {0xab, 0x07},
{0x24, 0x95},       {0x25, 0x33},
{0x26, 0xe3},       {0x9f, 0x78},
{0xa0, 0x68},       {0xa1, 0x03},
{0xa6, 0xd8},       {0xa7, 0xd8},
{0xa8, 0xf0},       {0xa9, 0x90},
{0xaa, 0x94},       {0x10, 0x00},

// AWB parameters
{0x43, 0x0a},       {0x44, 0xf0},
{0x45, 0x34},       {0x46, 0x58},
{0x47, 0x28},       {0x48, 0x3a},
{0x59, 0x88},       {0x5a, 0x88},
{0x5b, 0x44},       {0x5c, 0x67},
{0x5d, 0x49},       {0x5e, 0x0e},
{0x6c, 0x0a},       {0x6d, 0x55},
{0x6e, 0x11},       {0x6f, 0x9f},
{0x6a, 0x40},       {0x01, 0x40},
{0x02, 0x60},       {0x13, 0xe7},

// Additional parameters
{0x34, 0x11},       {0x3f, 0x00},
{0x75, 0x05},       {0x76, 0xe1},
{0x4c, 0x00},       {0x77, 0x01},
{0xb8, 0x0a},       {0x41, 0x18},
{0x3b, 0x12},       {0xa4, 0x88},
{0x96, 0x00},       {0x97, 0x30},
{0x98, 0x20},       {0x99, 0x30},
{0x9a, 0x84},       {0x9b, 0x29},
{0x9c, 0x03},       {0x9d, 0x4c},
{0x9e, 0x3f},       {0x78, 0x04},
{0x0e, 0x61},       {0x0f, 0x4b},
{0x16, 0x02},       {0x1e, 0x00},
{0x21, 0x02},       {0x22, 0x91},
{0x29, 0x07},       {0x33, 0x0b},
{0x35, 0x0b},       {0x37, 0x1d},
{0x38, 0x71},       {0x39, 0x2a},
{0x3c, 0x78},       {0x4d, 0x40},
{0x4e, 0x20},       {0x69, 0x00},
{0x6b, 0x3a},       {0x74, 0x10},
{0x8d, 0x4f},       {0x8e, 0x00},
{0x8f, 0x00},       {0x90, 0x00},
{0x91, 0x00},       {0x96, 0x00},
{0x9a, 0x00},       {0xb1, 0x0c},
{0xb2, 0x0e},       {0xb3, 0x82},
{0x4b, 0x01},
};

/* Initialization sequence for QQVGA resolution (160x120) */
const char OV7670_QQVGA[][2]=
{
   // Not implemented yet
  {0x12, 0x14},       //QVGA size, RGB mode
  {0x40, 0xd0},       //RGB565
  {0xb0, 0x84},       //Color mode
 
};

/**
  * @}
  */
  
/** @defgroup OV7670_Private_Functions
  * @{
  */ 
  
/**
  * @brief  Initializes the OV7670 CAMERA component.
  * @param  DeviceAddr: Device address on communication Bus.
  * @param  resolution: Camera resolution
  * @retval None
  */
void ov7670_Init(uint16_t DeviceAddr, uint32_t resolution)
{
  uint32_t index;
  
  /* Initialize I2C */
  CAMERA_IO_Init();    
  
  /* Prepare the camera to be configured by resetting all its registers */
  CAMERA_IO_Write(DeviceAddr, OV7670_SENSOR_COM7, 0x80);
  CAMERA_Delay(200);
  
  /* Initialize OV7670 */
  switch (resolution)
  {
  case CAMERA_R160x120:
    {
      for(index=0; index<(sizeof(OV7670_QQVGA)/2); index++)
      {
        CAMERA_IO_Write(DeviceAddr, OV7670_QQVGA[index][0], OV7670_QQVGA[index][1]);
        CAMERA_Delay(20);
      } 
      break;
    }    
  case CAMERA_R320x240:
    {
      for(index=0; index<(sizeof(OV7670_QVGA)/2); index++)
      {
        CAMERA_IO_Write(DeviceAddr, OV7670_QVGA[index][0], OV7670_QVGA[index][1]);
        CAMERA_Delay(20);
      } 
      break;
    }
  case CAMERA_R480x272:
    {
      /* Not supported resolution */
      break;
    }
  case CAMERA_R640x480:
    {
      for(index=0; index<(sizeof(OV7670_VGA)/2); index++)
      {
        CAMERA_IO_Write(DeviceAddr, OV7670_VGA[index][0], OV7670_VGA[index][1]);
        CAMERA_Delay(20);
      }
      break;
    }    
  default:
    {
      break;
    }
  }
}

/**
  * @brief  Configures the OV7670 camera feature.
  * @param  DeviceAddr: Device address on communication Bus.
  * @param  feature: Camera feature to be configured
  * @param  value: Value to be configured
  * @param  brightness_value: Brightness value to be configured
  * @retval None
  */
void ov7670_Config(uint16_t DeviceAddr, uint32_t feature, uint32_t value, uint32_t brightness_value)
{
  uint8_t tslb, mtx1, mtx2, mtx3, mtx4, mtx5, mtx6;
  uint64_t value_tmp;
  uint32_t br_value;
  
  /* Convert the input value into ov7670 parameters */
  value_tmp = ov7670_ConvertValue(feature, value); 
  br_value = (uint32_t)ov7670_ConvertValue(CAMERA_CONTRAST_BRIGHTNESS, brightness_value);
    
  switch(feature)
  {
  case CAMERA_CONTRAST_BRIGHTNESS:
    {
      CAMERA_IO_Write(DeviceAddr, OV7670_SENSOR_BRTN, br_value);
      CAMERA_IO_Write(DeviceAddr, OV7670_SENSOR_CNST1, value_tmp);
      break;
    }
  case CAMERA_BLACK_WHITE:
  case CAMERA_COLOR_EFFECT:
    {     
      tslb = (uint8_t)(value_tmp >> 48);
      mtx1 = (uint8_t)(value_tmp >> 40);
      mtx2 = (uint8_t)(value_tmp >> 32);
      mtx3 = (uint8_t)(value_tmp >> 24);
      mtx4 = (uint8_t)(value_tmp >> 16);
      mtx5 = (uint8_t)(value_tmp >> 8);
      mtx6 = (uint8_t)(value_tmp);
      CAMERA_IO_Write(DeviceAddr, OV7670_SENSOR_TSLB, tslb);
      CAMERA_IO_Write(DeviceAddr, OV7670_SENSOR_MTX1, mtx1);
      CAMERA_IO_Write(DeviceAddr, OV7670_SENSOR_MTX2, mtx2);
      CAMERA_IO_Write(DeviceAddr, OV7670_SENSOR_MTX3, mtx3);
      CAMERA_IO_Write(DeviceAddr, OV7670_SENSOR_MTX4, mtx4);
      CAMERA_IO_Write(DeviceAddr, OV7670_SENSOR_MTX5, mtx5);
      CAMERA_IO_Write(DeviceAddr, OV7670_SENSOR_MTX6, mtx6);
      break;
    }     
  default:
    {
      break;
    }
  }
}

/**
  * @brief  Read the OV7670 Camera identity.
  * @param  DeviceAddr: Device address on communication Bus.
  * @retval the OV7670 ID
  */
uint16_t ov7670_ReadID(uint16_t DeviceAddr)
{
  /* Initialize I2C */
  CAMERA_IO_Init();
  
  /* Get the camera ID */
  return (CAMERA_IO_Read(DeviceAddr, OV7670_SENSOR_PIDH));
}

/******************************************************************************
                            Static Functions
*******************************************************************************/
/**
  * @brief  Convert input values into ov7670 parameters.
  * @param  feature: Camera feature to be configured
  * @param  value: Value to be configured
  * @retval The converted value
  */
static uint64_t ov7670_ConvertValue(uint32_t feature, uint32_t value)
{
  uint64_t ret = 0;
  
  switch(feature)
  {
  case CAMERA_BLACK_WHITE:
    {
      switch(value)
      {
      case CAMERA_BLACK_WHITE_BW:
        {
          ret =  OV7670_BLACK_WHITE_BW;
          break;
        }
      case CAMERA_BLACK_WHITE_NEGATIVE:
        {
          ret =  OV7670_BLACK_WHITE_NEGATIVE;
          break;
        }
      case CAMERA_BLACK_WHITE_BW_NEGATIVE:
        {
          ret =  OV7670_BLACK_WHITE_BW_NEGATIVE;
          break;
        }
      case CAMERA_BLACK_WHITE_NORMAL:
        {
          ret =  OV7670_BLACK_WHITE_NORMAL;
          break;
        }
      default:
        {
          ret =  OV7670_BLACK_WHITE_NORMAL;
          break;
        }
      }
      break;
    }
  case CAMERA_CONTRAST_BRIGHTNESS:
    {
      switch(value)
      {
      case CAMERA_BRIGHTNESS_LEVEL0:
        {
          ret =  OV7670_BRIGHTNESS_LEVEL0;
          break;
        }
      case CAMERA_BRIGHTNESS_LEVEL1:
        {
          ret =  OV7670_BRIGHTNESS_LEVEL1;
          break;
        }
      case CAMERA_BRIGHTNESS_LEVEL2:
        {
          ret =  OV7670_BRIGHTNESS_LEVEL2;
          break;
        }
      case CAMERA_BRIGHTNESS_LEVEL3:
        {
          ret =  OV7670_BRIGHTNESS_LEVEL3;
          break;
        }
      case CAMERA_BRIGHTNESS_LEVEL4:
        {
          ret =  OV7670_BRIGHTNESS_LEVEL4;
          break;
        }        
      case CAMERA_CONTRAST_LEVEL0:
        {
          ret =  OV7670_CONTRAST_LEVEL0;
          break;
        }
      case CAMERA_CONTRAST_LEVEL1:
        {
          ret =  OV7670_CONTRAST_LEVEL1;
          break;
        }
      case CAMERA_CONTRAST_LEVEL2:
        {
          ret =  OV7670_CONTRAST_LEVEL2;
          break;
        }
      case CAMERA_CONTRAST_LEVEL3:
        {
          ret =  OV7670_CONTRAST_LEVEL3;
          break;
        }
      case CAMERA_CONTRAST_LEVEL4:
        {
          ret =  OV7670_CONTRAST_LEVEL4;
          break;
        }
      default:
        {
          ret =  OV7670_CONTRAST_LEVEL0;
          break;
        }
      }
      break;
    }
  case CAMERA_COLOR_EFFECT:
    {
      switch(value)
      {
      case CAMERA_COLOR_EFFECT_ANTIQUE:
        {
          ret =  OV7670_COLOR_EFFECT_ANTIQUE;
          break;
        }
      case CAMERA_COLOR_EFFECT_BLUE:
        {
          ret =  OV7670_COLOR_EFFECT_BLUE;
          break;
        }
      case CAMERA_COLOR_EFFECT_GREEN:
        {
          ret =  OV7670_COLOR_EFFECT_GREEN;
          break;
        }
      case CAMERA_COLOR_EFFECT_RED:
        {
          ret =  OV7670_COLOR_EFFECT_RED;
          break;
        }
      case CAMERA_COLOR_EFFECT_NONE:
      default:
        {
          ret =  OV7670_COLOR_EFFECT_NONE;
          break;
        }
      }
      break;
    default:
      {
        ret = 0;
        break;
      }    
    }
  }
  
  return ret;
}
         
/**
  * @}
  */ 
  
/**
  * @}
  */ 

/**
  * @}
  */ 
  
/**
  * @}
  */  

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
