/* Preface:
   This is the HeartRateService from the mbed Microcontroller library. 
   Copied from https://github.com/ARMmbed/mbed-os/blob/master/features/FEATURE_BLE/ble/services/HeartRateService.h
   (2259e0d)
   
   The source has not been changed, but comments have been added to help clarify:
      Behavior, 
      C++ Syntax, and
      Design decisions

Added comments are all in-line comments that start with:
  // COMMENT:       
      

*/

/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __BLE_HEART_RATE_SERVICE_H__
#define __BLE_HEART_RATE_SERVICE_H__

#include "ble/BLE.h"

/**
* @class HeartRateService
* @brief BLE Service for HeartRate. This BLE Service contains the location of the sensor and the heart rate in beats per minute.
* Service:  https://developer.bluetooth.org/gatt/services/Pages/ServiceViewer.aspx?u=org.bluetooth.service.heart_rate.xml
* HRM Char: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.heart_rate_measurement.xml
* Location: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.body_sensor_location.xml
*/

// COMMENT: It would be wise to review the definition of the HeartRateService
// COMMENT:    https://developer.bluetooth.org/gatt/services/Pages/ServiceViewer.aspx?u=org.bluetooth.service.heart_rate.xml
// COMMENT: Note that it contains three characteristics:  The Heart Rate Measurement, the Body Sensor Location, and the Heart Rate Cotrol Point.
class HeartRateService {

// COMMENT: C++ uses the labels public: , private: , and protected:  to control scope. 
// COMMENT: The scope applies to everything until the next scope label.    
public:
    /**
    * @enum SensorLocation
    * @brief Location of the heart rate sensor on body.
    */
    enum {
        LOCATION_OTHER = 0, /*!< Other location. */
        LOCATION_CHEST,     /*!< Chest. */
        LOCATION_WRIST,     /*!< Wrist. */
        LOCATION_FINGER,    /*!< Finger. */
        LOCATION_HAND,      /*!< Hand. */
        LOCATION_EAR_LOBE,  /*!< Earlobe. */
        LOCATION_FOOT,      /*!< Foot. */
    };

public:
    /**
     * @brief Constructor with 8-bit HRM Counter value.
     *
     * @param[ref] _ble
     *               Reference to the underlying BLE.
     * @param[in] hrmCounter (8-bit)
     *               Initial value for the HRM counter.
     * @param[in] location
     *               Sensor's location.
     */
    HeartRateService(BLE &_ble, uint8_t hrmCounter, uint8_t location) :
// COMMENT: This declares a constructor. The "BLE &_ble" parameter is an reference variable.
// COMMENT: Reference variables are a way to efficiently pass and use an object while limiting the way 
// COMMENT: it can be changed.  See: https://en.wikipedia.org/wiki/Reference_(C%2B%2B) for more.

// COMMENT: The lines below are a way to efficiently initialize the fields (aka instance variables) of the
// COMMENT: object.  In Java this is often done with several assignments in the body of the constructor.
// COMMENT:    Ex:  "valueBytes = hrmCounter;" within the function rather than "valueBytes(hrmCounter)" before
// COMMENT: the function begins.  (The behavior of the approach used here can be subtly different with C++ 
// COMMENT: objects and references)
// COMMENT: See: http://www.cprogramming.com/tutorial/initialization-lists-c++.html for more on initialization lists
        ble(_ble),
        valueBytes(hrmCounter),
// COMMENT: The hrmRate is declared as a "GattCharacteristic", which doesn't have any properties by default.
// COMMENT: Notice that READ and NOTIFY properties are being passed into the constructor to allow read/notify.
        hrmRate(GattCharacteristic::UUID_HEART_RATE_MEASUREMENT_CHAR, valueBytes.getPointer(),
                valueBytes.getNumValueBytes(), HeartRateValueBytes::MAX_VALUE_BYTES,
                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        hrmLocation(GattCharacteristic::UUID_BODY_SENSOR_LOCATION_CHAR, &location),
        controlPoint(GattCharacteristic::UUID_HEART_RATE_CONTROL_POINT_CHAR, &controlPointValue) {
// COMMENT: This is the body of the method. It just calls a private method to complete the setup.
// COMMENT: This was done so all constructors could use this same method for common setup steps
// COMMENT: (rather than replicate the code in each constructor) 
        setupService();
    }

    /**
     * @brief Constructor with a 16-bit HRM Counter value.
     *
     * @param[in] _ble
     *               Reference to the underlying BLE.
     * @param[in] hrmCounter (8-bit)
     *               Initial value for the HRM counter.
     * @param[in] location
     *               Sensor's location.
     */
// COMMENT: Another constructor.
    HeartRateService(BLE &_ble, uint16_t hrmCounter, uint8_t location) :
        ble(_ble),
        valueBytes(hrmCounter),
        hrmRate(GattCharacteristic::UUID_HEART_RATE_MEASUREMENT_CHAR, valueBytes.getPointer(),
                valueBytes.getNumValueBytes(), HeartRateValueBytes::MAX_VALUE_BYTES,
                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        hrmLocation(GattCharacteristic::UUID_BODY_SENSOR_LOCATION_CHAR, &location),
        controlPoint(GattCharacteristic::UUID_HEART_RATE_CONTROL_POINT_CHAR, &controlPointValue) {
        setupService();
    }

    /**
     * @brief Set a new 8-bit value for the heart rate.
     *
     * @param[in] hrmCounter
     *                  Heart rate in BPM.
     */
    void updateHeartRate(uint8_t hrmCounter) {
// COMMENT: This is method that will be used to update the value of the heart rate characteristic.
// COMMENT: valueBytes is the actual characteristic.  The following line of code will update its value.
        valueBytes.updateHeartRate(hrmCounter);
// COMMENT: But valueBytes is just the member variable in this object for the heart rate. The BLE subsystem
// COMMENT: needs to be notified that it has changed (so a notification can be sent)
// COMMENT: This is changing an "Attribute's value".  The BLE Server has it's own internal copy of all the
// COMMENT: data.  The following line of code will update this internal copy:
        ble.gattServer().write(hrmRate.getValueHandle(), valueBytes.getPointer(), valueBytes.getNumValueBytes());
// COMMENT: I.e., the server write to the heart rate characteristic's "value" by using the designated number of bytes.
// COMMENT: "pointers" are used to copy the data. For more on pointers see: https://en.wikibooks.org/wiki/C_Programming/Pointers_and_arrays
    }

    /**
     * Set a new 16-bit value for the heart rate.
     *
     * @param[in] hrmCounter
     *                  Heart rate in BPM.
     */
    void updateHeartRate(uint16_t hrmCounter) {
// COMMENT: Like the above, but using a 16-bit value instead.
        valueBytes.updateHeartRate(hrmCounter);
        ble.gattServer().write(hrmRate.getValueHandle(), valueBytes.getPointer(), valueBytes.getNumValueBytes());
    }

    /**
     * This callback allows the heart rate service to receive updates to the
     * controlPoint characteristic.
     *
     * @param[in] params
     *     Information about the characterisitc being updated.
     */
    virtual void onDataWritten(const GattWriteCallbackParams *params) {
        if (params->handle == controlPoint.getValueAttribute().getHandle()) {
// COMMENT: This is a virtual function --- It could be "overridden" by a subclass.
// COMMENT: The if-statement would allow this code to determine if a change had been 
// COMMENT: to the control point (the code to react to that change could be placed here) 
// COMMENT: This callback must be "connected" to the BLE stack (this is done in the setupService())
            /* Do something here if the new value is 1; else you can override this method by
             * extending this class.
             * @NOTE: If you are extending this class, be sure to also call
             * ble.onDataWritten(this, &ExtendedHRService::onDataWritten); in
             * your constructor.
             */
        }
    }

protected:
    void setupService(void) {
// COMMENT: This builds the service table (an array of the individual characteristics)
// COMMENT: Note the use of the "address-of" operation (&) to build the table. See: http://www.cplusplus.com/doc/tutorial/pointers/
        GattCharacteristic *charTable[] = {&hrmRate, &hrmLocation, &controlPoint};
        GattService         hrmService(GattService::UUID_HEART_RATE_SERVICE, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));

        ble.addService(hrmService);
// COMMENT: This setups up the "onDataWritten" callback above. I.e., it adds in a way to 
// COMMENT: respond when data is written to the BLE stack from the client.
        ble.onDataWritten(this, &HeartRateService::onDataWritten);
    }

protected:
    /* Private internal representation for the bytes used to work with the value of the heart rate characteristic. */
// COMMENT: This "struct" is like having another class nested (with all public member fields/methods)
// COMMENT: The Heart Rate Service can use two different formats for the heart rate data. This hides
// COMMENT: the details of which is being used. 
    struct HeartRateValueBytes {
        static const unsigned MAX_VALUE_BYTES  = 3; /* Flags, and up to two bytes for heart rate. */
        static const unsigned FLAGS_BYTE_INDEX = 0;

        static const unsigned VALUE_FORMAT_BITNUM = 0;
        static const uint8_t  VALUE_FORMAT_FLAG   = (1 << VALUE_FORMAT_BITNUM);

// The following two constructors allow objects to be constructed from either 8-bit or 16-bit values.
        HeartRateValueBytes(uint8_t hrmCounter) : valueBytes() {
            updateHeartRate(hrmCounter);
        }

        HeartRateValueBytes(uint16_t hrmCounter) : valueBytes() {
            updateHeartRate(hrmCounter);
        }

// COMMENT: This will allow an update of the data from an 8-bit value (for the 8-bit format)
        void updateHeartRate(uint8_t hrmCounter) {
            valueBytes[FLAGS_BYTE_INDEX]    &= ~VALUE_FORMAT_FLAG;
            valueBytes[FLAGS_BYTE_INDEX + 1] = hrmCounter;
        }

// COMMENT: This will allow an update of the data from an 16-bit value (for the 16-bit format)
        void updateHeartRate(uint16_t hrmCounter) {
            valueBytes[FLAGS_BYTE_INDEX]    |= VALUE_FORMAT_FLAG;
            valueBytes[FLAGS_BYTE_INDEX + 1] = (uint8_t)(hrmCounter & 0xFF);
            valueBytes[FLAGS_BYTE_INDEX + 2] = (uint8_t)(hrmCounter >> 8);
        }
// COMMENT: The two "getPointer()" functions allow access to the stored heart rate data via pointers.
// COMMENT: See: https://en.wikibooks.org/wiki/C_Programming/Pointers_and_arrays for details of pointers
        uint8_t       *getPointer(void) {
            return valueBytes;
        }

        const uint8_t *getPointer(void) const {
            return valueBytes;
        }

// COMMENT: The following getter will get the actual bytes promoted to a 32-bit unsigned. 
        unsigned       getNumValueBytes(void) const {
            return 1 + ((valueBytes[FLAGS_BYTE_INDEX] & VALUE_FORMAT_FLAG) ? sizeof(uint16_t) : sizeof(uint8_t));
        }

    private:
        /* First byte: 8-bit values, no extra info. Second byte: uint8_t HRM value */
        /* See https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.heart_rate_measurement.xml */
// COMMENT: The actual bytes are protected/hidden from direct access. 
        uint8_t valueBytes[MAX_VALUE_BYTES];
    };

protected:
// COMMENT: These are the fields (aka properties aka instance variables) of the object.
// COMMENT: The ble variable is used to access the entire BLE subsystem (GAP and GATT layers of the stack)
    BLE                 &ble;

// COMMENT: Use a "HeartRateValueBytes" object to keep track of the heart rate
// COMMENT: (This hide the details of how it is actually stored)
    HeartRateValueBytes  valueBytes;
// COMMENT: the controlPointValue represents the value of the control point.
    uint8_t              controlPointValue;

// COMMENT: The following three variables declare the actual characteristics
    GattCharacteristic                   hrmRate;
    ReadOnlyGattCharacteristic<uint8_t>  hrmLocation;
// COMMENT: The ReadOnlyGattCharacteristic uses templates (like Java's Generics) to specify the data type that will be used.
// COMMENT: In these cases a single byte will be used for both the hrmLocation and controlPoint characteristics.
    WriteOnlyGattCharacteristic<uint8_t> controlPoint;
    
// COMMENT: Notice that characteristics were declared with three different types:
// COMMENT:   Just "GattCharacteristic", which needs to have the data & properties specified in the constructor.
// COMMENT:   As "ReadOnlyGattCharacteristic", which will be readable (only)
// COMMENT:   And as "WriteOnlyGattCharacteristic", which will be writeabel (only) 
};

#endif /* #ifndef __BLE_HEART_RATE_SERVICE_H__*/