/* mbed Microcontroller Library
 * Copyright (c) 2017 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#include <events/mbed_events.h>
#include <stdio.h>
#include "mbed.h"

#include "platform/Callback.h"
#include "events/EventQueue.h"
#include "platform/NonCopyable.h"

#include "ble/BLE.h"
#include "ble/Gap.h"
#include "ble/GattClient.h"
#include "ble/GapAdvertisingParams.h"
#include "ble/GapAdvertisingData.h"
#include "ble/GattServer.h"
#include "BLEProcess.h"
#include "Si1133.h"


using mbed::callback;
        Si1133 sensor_light(P0_13, P0_15);


/**
 * A My service that demonstrate the GattServer features.
 *
 * The My service host three characteristics that model the current hour,
 * led and count of the My. The value of the count characteristic is
 * incremented automatically by the system.
 *
 * A client can subscribe to updates of the My characteristics and get
 * notified when one of the value is changed. Clients can also change value of
 * the count, led and hour characteristric.
 */
class MyService {
    typedef MyService Self;

public:
    MyService() :
       // _hour_char("485f4145-52b9-4644-af1f-7a6b9322490f", 0),
        _led_char("0a924ca7-87cd-4699-a3bd-abdcd9cf126a", 0),
        _count_char("8dd6a1b7-bc75-4741-8a26-264af75807de", 0),
        _My_service(
            /* uuid */ "51311102-030e-485f-b122-f8f381aa84ed",
            /* characteristics */ _My_characteristics,
            /* numCharacteristics */ sizeof(_My_characteristics) /
                                     sizeof(_My_characteristics[0])
        ),
        _server(NULL),
        _event_queue(NULL),
        _alive_led(P1_13, 1),
        _actuated_led(P1_14,0)
        
    {
        // update internal pointers (value, descriptors and characteristics array)
       // _My_characteristics[0] = &_hour_char;
        _My_characteristics[1] = &_led_char;
        _My_characteristics[0] = &_count_char;

        // setup authorization handlers
        //_hour_char.setWriteAuthorizationCallback(this, &Self::authorize_client_write);
        _led_char.setWriteAuthorizationCallback(this, &Self::authorize_client_write);
        _count_char.setWriteAuthorizationCallback(this, &Self::authorize_client_write);
    }



    void start(BLE &ble_interface, events::EventQueue &event_queue)
    {
         if (_event_queue) {
            return;
        }

        _server = &ble_interface.gattServer();
        _event_queue = &event_queue;


        // register the service
        printf("Adding demo service\r\n");
        ble_error_t err = _server->addService(_My_service);

        if (err) {
            printf("Error %u during demo service registration.\r\n", err);
            return;
        }

        // read write handler
        _server->onDataSent(as_cb(&Self::when_data_sent));
        _server->onDataWritten(as_cb(&Self::when_data_written));
        _server->onDataRead(as_cb(&Self::when_data_read));

        // updates subscribtion handlers
        _server->onUpdatesEnabled(as_cb(&Self::when_update_enabled));
        _server->onUpdatesDisabled(as_cb(&Self::when_update_disabled));
        _server->onConfirmationReceived(as_cb(&Self::when_confirmation_received));

        // print the handles
        printf("My service registered\r\n");
        printf("service handle: %u\r\n", _My_service.getHandle());
        //printf("\thour characteristic value handle %u\r\n", _hour_char.getValueHandle());
       // printf("\tled characteristic value handle %u\r\n", _led_char.getValueHandle());
        printf("\tcount characteristic value handle %u\r\n", _count_char.getValueHandle());

        _event_queue->call_every(1000 /* ms */, callback(this, &Self::increment_count));
        _event_queue->call_every(500 /* ms */, callback(this, &Self::blink));
        
    }

private:

    /**
     * Handler called when a notification or an indication has been sent.
     */
    void when_data_sent(unsigned count)
    {
        printf("sent %u updates\r\n", count);
    }

    /**
     * Handler called after an attribute has been written.
     */
    void when_data_written(const GattWriteCallbackParams *e)
    {
        printf("data written:\r\n");
        printf("\tconnection handle: %u\r\n", e->connHandle);
        printf("\tattribute handle: %u", e->handle);
        //if (e->handle == _hour_char.getValueHandle()) {
          //  printf(" (hour characteristic)\r\n");
       // } else
        if (e->handle == _led_char.getValueHandle()) {
            printf(" (led characteristic)\r\n");
            _actuated_led = *(e->data);
        } else if (e->handle == _count_char.getValueHandle()) {
            printf(" (count characteristic)\r\n");
        } else {
            printf("\r\n");
        }
        printf("\twrite operation: %u\r\n", e->writeOp);
        printf("\toffset: %u\r\n", e->offset);
        printf("\tlength: %u\r\n", e->len);
        printf("\t data: ");

        for (size_t i = 0; i < e->len; ++i) {
            printf("%02X", e->data[i]);
        }

        printf("\r\n");
    }

    /**
     * Handler called after an attribute has been read.
     */
    void when_data_read(const GattReadCallbackParams *e)
    {
        printf("data read:\r\n");
        printf("\tconnection handle: %u\r\n", e->connHandle);
        printf("\tattribute handle: %u", e->handle);
       // if (e->handle == _hour_char.getValueHandle()) {
       //     printf(" (hour characteristic)\r\n");
       // } else 
       if (e->handle == _led_char.getValueHandle()) {
            printf(" (led characteristic)\r\n");
        } else if (e->handle == _count_char.getValueHandle()) {
            printf(" (count characteristic)\r\n");
        } else {
            printf("\r\n");
        }
    }

    /**
     * Handler called after a client has subscribed to notification or indication.
     *
     * @param handle Handle of the characteristic value affected by the change.
     */
    void when_update_enabled(GattAttribute::Handle_t handle)
    {
        printf("update enabled on handle %d\r\n", handle);
    }

    /**
     * Handler called after a client has cancelled his subscription from
     * notification or indication.
     *
     * @param handle Handle of the characteristic value affected by the change.
     */
    void when_update_disabled(GattAttribute::Handle_t handle)
    {
        printf("update disabled on handle %d\r\n", handle);
    }

    /**
     * Handler called when an indication confirmation has been received.
     *
     * @param handle Handle of the characteristic value that has emitted the
     * indication.
     */
    void when_confirmation_received(GattAttribute::Handle_t handle)
    {
        printf("confirmation received on handle %d\r\n", handle);
    }

    /**
     * Handler called when a write request is received.
     *
     * This handler verify that the value submitted by the client is valid before
     * authorizing the operation.
     */
    void authorize_client_write(GattWriteAuthCallbackParams *e)
    {
        printf("characteristic %u write authorization\r\n", e->handle);

        if (e->offset != 0) {
            printf("Error invalid offset\r\n");
            e->authorizationReply = AUTH_CALLBACK_REPLY_ATTERR_INVALID_OFFSET;
            return;
        }

        if (e->len != 1) {
            printf("Error invalid len\r\n");
            e->authorizationReply = AUTH_CALLBACK_REPLY_ATTERR_INVALID_ATT_VAL_LENGTH;
            return;
        }

        //if ((e->data[0] >= 60) ||
          //  ((e->data[0] >= 24) && (e->handle == _hour_char.getValueHandle()))) {
         //   printf("Error invalid data\r\n");
         //   e->authorizationReply = AUTH_CALLBACK_REPLY_ATTERR_WRITE_NOT_PERMITTED;
         //   return;
       // }

        e->authorizationReply = AUTH_CALLBACK_REPLY_SUCCESS;
    }

    /**
     * Increment the count counter.
     */
    void increment_count(void)
    {
        uint8_t count =0;
        float float32;
        uint8_t count_v[4];
        uint8_t *vp;
        uint8_t temp;
        

        ble_error_t err = _count_char.get(*_server, count);
        if (err) {
            printf("read of the count value returned error %u\r\n", err);
            return;
        }
        //   count ++;
           
          
          if (sensor_light.open()) {
        printf("Device detected!\n");
            //Print the current light level
            printf("Lux = %.3f\n", (float)sensor_light.get_light_level());
            //Print the current UV index
            printf("UV index = %.3f\n", (float)sensor_light.get_uv_index());
            //Sleep for 0.5 seconds
            float32= sensor_light.get_light_level();
            count = (uint8_t)float32;
            //printf("Lux = %.2f\n", float32);
            vp = (uint8_t*)&float32;
            count_v[0] = vp[0];
            count_v[1] = vp[1];
            count_v[2] = vp[2];
            count_v[3] = vp[3];
            *(float*)count_v = float32;
            count = count_v[2];
            
            _actuated_led = !_actuated_led;
        }

        err = _count_char.set(*_server, count);
        if (err) {
            printf("write of the count value returned error %u\r\n", err);
            return;
        }

    }
    
    void blink() {
        _alive_led = !_alive_led;
    }

    /**
     * Change led status.
     */
    void change_led(void)
    {
        uint8_t led = 0;
        ble_error_t err = _led_char.get(*_server, led);
        if (err) {
            printf("read of the led value returned error %u\r\n", err);
            return;
        }
    }

   /*  * Increment the hour counter.

    void increment_hour(void)
    {
        uint8_t hour = 0;
        ble_error_t err = _hour_char.get(*_server, hour);
        if (err) {
            printf("read of the hour value returned error %u\r\n", err);
            return;
        }

        hour = (hour + 1) % 24; 

        err = _hour_char.set(*_server, hour);
        if (err) {
            printf("write of the hour value returned error %u\r\n", err);
            return;
        }
    }
*/
private:
    /**
     * Helper that construct an event handler from a member function of this
     * instance.
     */
    template<typename Arg>
    FunctionPointerWithContext<Arg> as_cb(void (Self::*member)(Arg))
    {
        return makeFunctionPointer(this, member);
    }

    /**
     * Read, Write, Notify, Indicate  Characteristic declaration helper.
     *
     * @tparam T type of data held by the characteristic.
     */
    template<typename T>
    class ReadWriteNotifyIndicateCharacteristic : public GattCharacteristic {
    public:
        /**
         * Construct a characteristic that can be read or written and emit
         * notification or indication.
         *
         * @param[in] uuid The UUID of the characteristic.
         * @param[in] initial_value Initial value contained by the characteristic.
         */
        ReadWriteNotifyIndicateCharacteristic(const UUID & uuid, const T& initial_value) :
            GattCharacteristic(
                /* UUID */ uuid,
                /* Initial value */ &_value,
                /* Value size */ sizeof(_value),
                /* Value capacity */ sizeof(_value),
                /* Properties */ GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ |
                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE |
                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY |
                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_INDICATE,
                /* Descriptors */ NULL,
                /* Num descriptors */ 0,
                /* variable len */ false
            ),
            _value(initial_value) {
        }

        /**
         * Get the value of this characteristic. 
         *
         * @param[in] server GattServer instance that contain the characteristic
         * value.
         * @param[in] dst Variable that will receive the characteristic value.
         *
         * @return BLE_ERROR_NONE in case of success or an appropriate error code.
         */
        ble_error_t get(GattServer &server, T& dst) const
        {
            uint16_t value_length = sizeof(dst);
            return server.read(getValueHandle(), &dst, &value_length);
        }

        /**
         * Assign a new value to this characteristic.
         *
         * @param[in] server GattServer instance that will receive the new value.
         * @param[in] value The new value to set.
         * @param[in] local_only Flag that determine if the change should be kept
         * locally or forwarded to subscribed clients.
         */
          ble_error_t set(
               GattServer &server, const uint8_t &value, bool local_only = false
        ) const {
            return server.write(getValueHandle(), &value, sizeof(value), local_only);
        }

    private:
        uint8_t _value;
    };

    //ReadWriteNotifyIndicateCharacteristic<uint8_t> _hour_char;
    ReadWriteNotifyIndicateCharacteristic<uint8_t> _led_char;
    ReadWriteNotifyIndicateCharacteristic<uint8_t> _count_char;

    // list of the characteristics of the My service
    GattCharacteristic* _My_characteristics[2];

    // demo service
    GattService _My_service;

    GattServer* _server;
    events::EventQueue *_event_queue;
    
   DigitalOut _alive_led;
   DigitalOut _actuated_led;
};

int main() {
    BLE &ble_interface = BLE::Instance();
    events::EventQueue event_queue;
    MyService demo_service;
    BLEProcess ble_process(event_queue, ble_interface);
    

    ble_process.on_init(callback(&demo_service, &MyService::start));


    // bind the event queue to the ble interface, initialize the interface
    // and start advertising
    ble_process.start();

    // Process the event queue.
    event_queue.dispatch_forever();

    return 0;
}
