#include "mbed.h"
#include "EthernetInterface.h"
#include "nsdl_support.h"
#include "dbg.h"
#include "buffered_rgb_resource.h"

Serial pc(USBTX, USBRX); // tx, rx

// ****************************************************************************
// Configuration section

// Ethernet configuration
/* Define this to enable DHCP, otherwise manual address configuration is used */
#define DHCP

/* Manual IP configurations, if DHCP not defined */
#define IP      "10.45.0.206"
#define MASK    "255.255.255.0"
#define GW      "10.45.0.1"

// NSP configuration
/* Change this IP address to that of your NanoService Platform installation */
static const char* NSP_ADDRESS = "barista-mds.cloudapp.net";
static const int NSP_PORT = 5683;
char endpoint_name[] = "nespresso-client";
uint8_t ep_type[] = {"mbed_device"};
uint8_t lifetime_ptr[] = {"1200"};

// ****************************************************************************
// Ethernet initialization

EthernetInterface eth;

static void ethernet_init()
{
    /* Initialize network */
#ifdef DHCP
    NSDL_DEBUG("DHCP in use\r\n");
    eth.init();
#else
    eth.init(IP, MASK, GW);
#endif
    if(eth.connect(30000) == 0) 
    {
        pc.printf("Connect OK\n\r");
    } 

    NSDL_DEBUG("IP Address:%s ", eth.getIPAddress());
}

// ****************************************************************************
// NSP initialization

UDPSocket server;
Endpoint nsp;

static void nsp_init()
{
    server.init();
    server.bind(NSP_PORT);

    nsp.set_address(NSP_ADDRESS, NSP_PORT);
    
    NSDL_DEBUG("name: %s", endpoint_name);
    NSDL_DEBUG("NSP=%s - port %d\n", NSP_ADDRESS, NSP_PORT);
}

// ****************************************************************************
// Resource creation

static int create_resources()
{
    sn_nsdl_resource_info_s *resource_ptr = NULL;
    sn_nsdl_ep_parameters_s *endpoint_ptr = NULL;
    
    NSDL_DEBUG("Creating resources");

    /* Create resources */
    resource_ptr = (sn_nsdl_resource_info_s*)nsdl_alloc(sizeof(sn_nsdl_resource_info_s));
    if(!resource_ptr)
        return 0;
    memset(resource_ptr, 0, sizeof(sn_nsdl_resource_info_s));

    resource_ptr->resource_parameters_ptr = (sn_nsdl_resource_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_resource_parameters_s));
    if(!resource_ptr->resource_parameters_ptr)
    {
        nsdl_free(resource_ptr);
        return 0;
    }
    memset(resource_ptr->resource_parameters_ptr, 0, sizeof(sn_nsdl_resource_parameters_s));

    //create_nespresso_resource(resource_ptr);
    create_buffered_rgb_resource(resource_ptr);

    /* Register with NSP */
    endpoint_ptr = nsdl_init_register_endpoint(endpoint_ptr, (uint8_t*)endpoint_name, ep_type, lifetime_ptr);
    if(sn_nsdl_register_endpoint(endpoint_ptr) != 0)
        pc.printf("NSP registering failed\r\n");
    else
        pc.printf("NSP registering OK\r\n");
    nsdl_clean_register_endpoint(&endpoint_ptr);

    nsdl_free(resource_ptr->resource_parameters_ptr);
    nsdl_free(resource_ptr);
    return 1;
}

// ****************************************************************************
// Program entry point

extern "C" void mbed_mac_address(char *mac){
    mac[0] = 0x00;
    mac[1] = 0x02;
    mac[2] = 0xF7;
    mac[3] = 0xF1;
    mac[4] = 0x91;
    mac[5] = 0x9F;
};

int main()
{
    pc.baud(115200);

    // Initialize Ethernet interface first
    pc.printf("Initializing ethernet... ");
    ethernet_init();
    pc.printf("done.\r\n");
        
    // Initialize NSP node
    pc.printf("Initializing nsp node... ");
    nsp_init();
    pc.printf("done.\r\n");
    
    // Initialize NSDL stack
    pc.printf("Initializing NSDL stack... ");    
    nsdl_init();
    pc.printf("done.\r\n");
    
    // Create NSDL resources
    pc.printf("Initializing resources... ");    
    create_resources();
    pc.printf("done.\r\n");
    
    // Run the NSDL event loop (never returns)
    nsdl_event_loop();
}
