/******************************************************************//**
* @file     BMP280.c
* @brief    Contains all functions support for I2C based BMP280
*           Pressure Sensor library
* @version  1.0
* @date     26. May. 2016
* @author   Bhavin.Edutech Learning Solutions
**********************************************************************/

#include "BMP280.h"

//******************************************************************************
BMP280::BMP280(PinName sda, PinName scl)
{
    i2c_ = new I2C(sda, scl);
    i2c_owner = true;

    i2c_->frequency(400000);
}

//******************************************************************************
BMP280::BMP280(I2C *i2c) :
    i2c_(i2c)
{
    i2c_owner = false;
}

//******************************************************************************
BMP280::~BMP280()
{
    if(i2c_owner) {
        delete i2c_;
    }
}

//******************************************************************************

void BMP280::writeByte(uint8_t Address, uint8_t data)
{
   char data_write[2];
   data_write[0] = Address;
   data_write[1] = data;
   i2c_->write(BMP280_I2C_ADDRESS2, data_write, 2, 0);
}

//******************************************************************************

char BMP280::readByte(uint8_t Address)
{
    char data[1]; // `data` will store the register data     
    char data_write[1];
    data_write[0] = Address;
    i2c_->write(BMP280_I2C_ADDRESS2, data_write, 1, 1); // no stop
    i2c_->read(BMP280_I2C_ADDRESS2, data, 1, 0); 
    return data[0]; 
}

//******************************************************************************

void BMP280::readBytes(uint8_t Address, uint8_t count, uint8_t * dest)
{     
    char data[14];
    char data_write[1];
    data_write[0] = Address;
    i2c_->write(BMP280_I2C_ADDRESS2, data_write, 1, 1); // no stop
    i2c_->read(BMP280_I2C_ADDRESS2, data, count, 0); 
    for(int ii = 0; ii < count; ii++) {
     dest[ii] = data[ii];
    }
} 

//******************************************************************************
uint8_t BMP280::BMP280_GetID(void)
{
  return readByte(BMP280_CHIP_ID_REG);
}
//******************************************************************************
void BMP280::BMP280_SoftReset(void)
{
  writeByte(BMP280_RST_REG, BMP280_SOFT_RESET_CODE);
}
//******************************************************************************
uint8_t BMP280::BMP280_GetStatus(void)
{
  return readByte(BMP280_STAT_REG);
}
//******************************************************************************
uint8_t BMP280::BMP280_GetCtrlMeasurement(void)
{
  return readByte(BMP280_CTRL_MEAS_REG);
}
//******************************************************************************
uint8_t BMP280::BMP280_GetConfig(void)
{
  return readByte(BMP280_CONFIG_REG);
}
//******************************************************************************
void BMP280::BMP280_ReadMeasurements(void)
{
    uint8_t rx_Buf[6]= {0};
  
    readBytes(BMP280_PRESSURE_MSB_REG, 6, &rx_Buf[0]);
    
    adc_t  = (uint32_t)rx_Buf[BMP280_DATA_FRAME_TEMPERATURE_XLSB_BYTE] >> 4;
    adc_t |= (uint32_t)rx_Buf[BMP280_DATA_FRAME_TEMPERATURE_LSB_BYTE] << 4;
    adc_t |= (uint32_t)rx_Buf[BMP280_DATA_FRAME_TEMPERATURE_MSB_BYTE] << 12;
    
    adc_p  = (uint32_t)rx_Buf[BMP280_DATA_FRAME_PRESSURE_XLSB_BYTE] >> 4;
    adc_p |= (uint32_t)rx_Buf[BMP280_DATA_FRAME_PRESSURE_LSB_BYTE] << 4;
    adc_p |= (uint32_t)rx_Buf[BMP280_DATA_FRAME_PRESSURE_MSB_BYTE] << 12;
}

//******************************************************************************

void BMP280::BMP280_ReadCalibrationParams(void)
{
   uint8_t lsb, msb;

   msb = readByte(BMP280_TEMPERATURE_CALIB_DIG_T1_MSB_REG);
   cal_param.dig_T1 = (uint16_t) msb;
   lsb = readByte(BMP280_TEMPERATURE_CALIB_DIG_T1_LSB_REG);
   cal_param.dig_T1 = (cal_param.dig_T1 << 8) + lsb;


   msb = readByte(BMP280_TEMPERATURE_CALIB_DIG_T2_MSB_REG);
   cal_param.dig_T2 = (int16_t) msb;
   lsb = readByte(BMP280_TEMPERATURE_CALIB_DIG_T2_LSB_REG);
   cal_param.dig_T2 = (cal_param.dig_T2 << 8) + lsb;


   msb = readByte(BMP280_TEMPERATURE_CALIB_DIG_T3_MSB_REG);
   cal_param.dig_T3 = (int16_t) msb;
   lsb = readByte(BMP280_TEMPERATURE_CALIB_DIG_T3_LSB_REG);
   cal_param.dig_T3 = (cal_param.dig_T3 << 8) + lsb;


   msb = readByte(BMP280_PRESSURE_CALIB_DIG_P1_MSB_REG);
   cal_param.dig_P1 = (uint16_t) msb;
   lsb = readByte(BMP280_PRESSURE_CALIB_DIG_P1_LSB_REG);
   cal_param.dig_P1 = (cal_param.dig_P1 << 8) + lsb;


   msb = readByte(BMP280_PRESSURE_CALIB_DIG_P2_MSB_REG);
   cal_param.dig_P2 = (int16_t) msb;
   lsb = readByte(BMP280_PRESSURE_CALIB_DIG_P2_LSB_REG);
   cal_param.dig_P2 = (cal_param.dig_P2 << 8) + lsb;


   msb = readByte(BMP280_PRESSURE_CALIB_DIG_P3_MSB_REG);
   cal_param.dig_P3 = (int16_t) msb;
   lsb = readByte(BMP280_PRESSURE_CALIB_DIG_P3_LSB_REG);
   cal_param.dig_P3 = (cal_param.dig_P3 << 8) + lsb;


   msb = readByte(BMP280_PRESSURE_CALIB_DIG_P4_MSB_REG);
   cal_param.dig_P4 = (int16_t) msb;
   lsb = readByte(BMP280_PRESSURE_CALIB_DIG_P4_LSB_REG);
   cal_param.dig_P4 = (cal_param.dig_P4 << 8) + lsb;


   msb = readByte(BMP280_PRESSURE_CALIB_DIG_P5_MSB_REG);
   cal_param.dig_P5 = (int16_t) msb;
   lsb = readByte(BMP280_PRESSURE_CALIB_DIG_P5_LSB_REG);
   cal_param.dig_P5 = (cal_param.dig_P5 << 8) + lsb;


   msb = readByte(BMP280_PRESSURE_CALIB_DIG_P6_MSB_REG);
   cal_param.dig_P6 = (int16_t) msb;
   lsb = readByte(BMP280_PRESSURE_CALIB_DIG_P6_LSB_REG);
   cal_param.dig_P6 = (cal_param.dig_P6 << 8) + lsb;


   msb = readByte(BMP280_PRESSURE_CALIB_DIG_P7_MSB_REG);
   cal_param.dig_P7 = (int16_t) msb;
   lsb = readByte(BMP280_PRESSURE_CALIB_DIG_P7_LSB_REG);
   cal_param.dig_P7 = (cal_param.dig_P7 << 8) + lsb;


   msb = readByte(BMP280_PRESSURE_CALIB_DIG_P8_MSB_REG);
   cal_param.dig_P8 = (int16_t) msb;
   lsb = readByte(BMP280_PRESSURE_CALIB_DIG_P8_LSB_REG);
   cal_param.dig_P8 = (cal_param.dig_P8 << 8) + lsb;


   msb = readByte(BMP280_PRESSURE_CALIB_DIG_P9_MSB_REG);
   cal_param.dig_P9 = (int16_t) msb;
   lsb = readByte(BMP280_PRESSURE_CALIB_DIG_P9_LSB_REG);
   cal_param.dig_P9 = (cal_param.dig_P9 << 8) + lsb;
}

//******************************************************************************

void BMP280::BMP280_SetOversamplingPressure(uint8_t Value)
{
  uint8_t ctrlm;

  ctrlm = BMP280_GetCtrlMeasurement();
  ctrlm &= ~BMP280_CTRL_MEAS_REG_OVERSAMP_PRESSURE__MSK;
  ctrlm |= Value << BMP280_CTRL_MEAS_REG_OVERSAMP_PRESSURE__POS;

  writeByte(BMP280_CTRL_MEAS_REG, ctrlm);
  ctrlm = BMP280_GetCtrlMeasurement();
}

//******************************************************************************

void BMP280::BMP280_SetOversamplingTemperature(uint8_t Value)
{
  int8_t ctrlm;

  ctrlm = BMP280_GetCtrlMeasurement();
  ctrlm &= ~BMP280_CTRL_MEAS_REG_OVERSAMP_TEMPERATURE__MSK;
  ctrlm |= Value << BMP280_CTRL_MEAS_REG_OVERSAMP_TEMPERATURE__POS;

  writeByte(BMP280_CTRL_MEAS_REG, ctrlm);
}

//******************************************************************************

void BMP280::BMP280_SetPowerMode(BMP280_Mode_Type Value)
{
  uint8_t ctrlm;

  ctrlm = BMP280_GetCtrlMeasurement();
  ctrlm |= Value;

  writeByte(BMP280_CTRL_MEAS_REG, ctrlm);
}

//******************************************************************************

void BMP280::BMP280_SetFilterCoefficient(BMP280_Filter_Coeff_Type Value)
{
  uint8_t cfgv;

  cfgv = BMP280_GetConfig();
  cfgv &= ~BMP280_CONFIG_REG_FILTER__MSK;
  cfgv |= Value << BMP280_CONFIG_REG_FILTER__POS;
}

//******************************************************************************

void BMP280::BMP280_SetStandbyTime(BMP280_Standby_Type Value)
{
  uint8_t cfgv;

  cfgv = BMP280_GetConfig();
  cfgv &= ~BMP280_CONFIG_REG_STANDBY_DURN__MSK;
  cfgv |= Value << BMP280_CONFIG_REG_STANDBY_DURN__POS;
}

//******************************************************************************

uint8_t BMP280::BMP280_IsMeasuring(void)
{
  uint8_t output;

  output = BMP280_GetStatus();
  return (output & BMP280_STATUS_REG_MEASURING__MSK);
}

//******************************************************************************

int32_t BMP280::BMP280_Compensate_T(void)
{
  int32_t temp1, temp2, T;

  temp1 = ((((adc_t>>3) -((int32_t)cal_param.dig_T1<<1))) * ((int32_t)cal_param.dig_T2)) >> 11;
  temp2 = (((((adc_t>>4) - ((int32_t)cal_param.dig_T1)) * ((adc_t>>4) - ((int32_t)cal_param.dig_T1))) >> 12) * ((int32_t)cal_param.dig_T3)) >> 14;
  t_fine = temp1 + temp2;
  T = (t_fine * 5 + 128) >> 8;
  return T;
}

//******************************************************************************

uint32_t BMP280::BMP280_Compensate_P()
{
  int32_t press1, press2;
  uint32_t P;

  press1 = (((int32_t)t_fine)>>1) - (int32_t)64000;
  press2 = (((press1>>2) * (press1>>2)) >> 11 ) * ((int32_t)cal_param.dig_P6);
  press2 = press2 + ((press1*((int32_t)cal_param.dig_P5))<<1);
  press2 = (press2>>2)+(((int32_t)cal_param.dig_P4)<<16);
  press1 = (((cal_param.dig_P3 * (((press1>>2) * (press1>>2)) >> 13 )) >> 3) + ((((int32_t)cal_param.dig_P2) * press1)>>1))>>18;
  press1 =((((32768+press1))*((int32_t)cal_param.dig_P1))>>15);
  if (press1 == 0)
  {
    return 0; // avoid exception caused by division by zero
  }
  P = (((uint32_t)(((int32_t)1048576)-adc_p)-(press2>>12)))*3125;
  if (P < 0x80000000)
  {
    P = (P << 1) / ((uint32_t)press1);
  }
  else
  {
    P = (P / (uint32_t)press1) * 2;
  }
  press1 = (((int32_t)cal_param.dig_P9) * ((int32_t)(((P>>3) * (P>>3))>>13)))>>12;
  press2 = (((int32_t)(P>>2)) * ((int32_t)cal_param.dig_P8))>>13;
  P = (uint32_t)((int32_t)P + ((press1 + press2 + cal_param.dig_P7) >> 4));
  return P;
}

//******************************************************************************

void BMP280::BMP280_INIT(void)
{
  BMP280_SetStandbyTime(BMP280_STANDBY_TIME_1_MS);                              // Standby time 1ms
  BMP280_SetFilterCoefficient(BMP280_FILTER_COEFF_16);                          // IIR Filter coefficient 16
  BMP280_SetOversamplingPressure(BMP280_OVERSAMP_16X);                          // Pressure x16 oversampling
  BMP280_SetOversamplingTemperature(BMP280_OVERSAMP_2X);                        // Temperature x2 oversampling
  BMP280_SetPowerMode(BMP280_NORMAL_MODE);
}

//******************************************************************************

float BMP280::BMP280_GetTemperature(void)
{
  return BMP280_Compensate_T();
}

//******************************************************************************

float BMP280::BMP280_GetPressure(void)
{
  return BMP280_Compensate_P();
}
