#include "mbed.h"
#include "nrf51.h"
#include "nrf51_bitfields.h"

#include "ble/BLE.h"
#include "JoystickService.h"
#include "DFUService.h"
#include "UARTService.h"

#include "examples_common.h"

#define LOG(...)    { pc.printf(__VA_ARGS__); }

#define LED_GREEN   p21
#define LED_RED     p22
#define LED_BLUE    p23

#define LED_OFF 1
#define LED_ON  0

#define UART_TX     p9
#define UART_RX     p11
#define UART_CTS    p8
#define UART_RTS    p10

#define DATA_TX p3
#define DATA_RX p6

DigitalOut blue(LED_BLUE);
DigitalOut green(LED_GREEN);
DigitalOut red(LED_RED);

Serial pc(UART_TX, UART_RX);
Serial data(DATA_TX, DATA_RX);

BLE  ble;
JoystickService *joystickServicePtr;
static const char DEVICE_NAME[] = "BunnyJoystick";
static const char SHORT_DEVICE_NAME[] = "joystick0";

volatile bool bleIsConnected = false;
volatile uint8_t tick_event = 0;

static void onDisconnect(const Gap::DisconnectionCallbackParams_t *params) {
    LOG("disconnected\r\n");
    bleIsConnected = false;
    red = LED_OFF;
    blue = LED_OFF;
    green = LED_ON;
    ble.gap().startAdvertising(); // restart advertising
}


static void onConnect(const Gap::ConnectionCallbackParams_t *params) {
    LOG("connected\r\n");
    bleIsConnected = true;
    red = LED_OFF;
    blue = LED_ON;
    green = LED_OFF;
}

static void waiting() {
    if (!joystickServicePtr->isConnected()) {
        green = !green;
    } else {
        blue = !blue;
    }
}


int main() {
    blue = LED_OFF;
    green = LED_OFF;
    red = LED_OFF;
    
    data.baud(115200);

    wait(4);
    LOG("Bunny Joystick started.\n");

    LOG("initialising ticker\r\n");
    Ticker heartbeat;
    heartbeat.attach(waiting, 1);

    LOG("initialising ble\r\n");
    ble.init();

    ble.gap().onDisconnection(onDisconnect);
    ble.gap().onConnection(onConnect);

    initializeSecurity(ble);

    LOG("adding hid service\r\n");
    JoystickService joystickService(ble);
    joystickServicePtr = &joystickService;

    LOG("adding dev info and battery service\r\n");
    initializeHOGP(ble);

    LOG("setting up gap\r\n");
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::JOYSTICK);
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::COMPLETE_LOCAL_NAME,
        (const uint8_t *)DEVICE_NAME,
        sizeof(DEVICE_NAME));
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::SHORTENED_LOCAL_NAME,
        (const uint8_t *)SHORT_DEVICE_NAME,
        sizeof(SHORT_DEVICE_NAME));
    ble.gap().setDeviceName((const uint8_t *)DEVICE_NAME);

    LOG("advertising\r\n");
    ble.gap().startAdvertising();
    
    int xraw, yraw, zraw;

    while (true) {
        if (data.readable()) {
            char buffer[256] = {0};
            char c;
            int i = 0;
            while(data.readable() && i < 256) {
                buffer[i++] = c = data.getc();
                if (c == '\n') {
                    buffer[i] = 0;
                    break;
                }
            }
            LOG("Received data from FTHR:\n");
            LOG(buffer);
            sscanf(buffer, "%d,%d,%d", &xraw, &yraw, &zraw);
            LOG("%d,%d,%d", xraw, yraw, zraw);
            joystickServicePtr->setSpeed(xraw, yraw, zraw);
            
            wait(0.5);
        }

        ble.waitForEvent();
    }
}
