// -------------------------------------------------------------------------- 
// Name : e_eon.h
//
//	 Definition of input/output interface between Epics and the control system or VISSIM, respectively.
//
//	 Author : J. Mertz / R. Hildebrandt
//
//	 Version 6.2 for PTV group
//
// -------------------------------------------------------------------------- 


#ifndef e_eon_h
#define e_eon_h


/* constants */
#define MAXDETID 255						/* maximum number detectors (might be reduced; then a constant MAXDET and an IntlDetIndex would be needed. Would be some effort.) */
#define MAXVAR 30							/* maximum number return variables */
#define MAXSG 96							/* maximum number signal groups; number from VS-PLUS */
#define MAXSGID 255							/* maximum signal-group ID under OCIT */
#define MAXMPPID 98							/* message-point pairs with ID > 98 are not accepted */
#define LENGTHLOGSTRING 1024				/* 500 was not enough for initializeLogfileQueues at 536 in Krakow */
#define LENGTHDEBUGLOGSTRING LENGTHLOGSTRING*2
#define MAXJSONMESSAGES 100					/* maximum number lines in json file in order to restrict its size */

// -------------------------------------------------------------------------- 

typedef char messageText_t[LENGTHLOGSTRING];

typedef struct
{
  unsigned short messageNumber;
  unsigned char importance;
  messageText_t messageText;
} message_t;

typedef struct                        // structure of input variables 
{
  short	callingMode;                  // calling mode (2 = initialization, 1 = program change, 0 = standard call, 3 = read new supply, 4 = activate new supply, -1 free memory)
  short program;                      // current or desired signal program 
  short tX;                           // current cycle second tX 
  short deltaT;                       // time span since last call of PTV Epics. Usually 1 second.
  short detOccupancy[MAXDETID + 1];   // array detector occupancy [%]
  short detDemand[MAXDETID + 1];      // array detector demand [0/1] (1 = rising shoulder) 
  short detFailure[MAXDETID + 1];     // array detector failure [0/1] (1 = failure) 
  short detPriority[MAXDETID + 1];    // array detector priority [0/1 (default)/2/3]; only for detectors that are part of a message-point pair
  short signalState[MAXSGID + 1];     // array signal state (1 = green, 2 = red, 8 = amber, 7 = red-amber, 11 = flashing green)
  short numberPTVehicles[MAXMPPID + 1];// array number vehicles within message-point pair; used for forced logoff; must be set to -1 if not used
  unsigned char language;             // language of messages; 1 = English, 4 = Polish; in TRENDS: 0 = German
  char nodeName[50];                  // usually only the number; not really necessary when supply is handed over; useful nonetheless since else all supply log files have the same name (there is only one for all intersections)
#ifdef RBC
  unsigned char	cycleLength;          // cycle length of current program; only for testing
  short	BalanceFSP[33];               // Balance frame signal plan; 0th entry: Offset; 1st-16th entry: motorized traffic; 17th-32th entry: pedestrians. 
#endif
#if defined VISSIM
  unsigned char balanceFixedTime;     // Epics is the slave of Balance, i.e. it is supposed to reproduce exactly the Balance signal plan; wre das nicht auch im Realbetrieb interessant?
  unsigned char writeLogFiles;        // is Epics supposed to write Log files? Controlled via the flag 'Debug mode' ('Ablaufverfolgung') of the signal control editor
#endif
#if defined UseInputdata || defined VISSIM || defined RBC
  char *netCommand;					  // net command containing BALANCE frame signal plan, parsed by EPICS
#endif
} EPICSInput_t;

typedef struct                        // structure of output variables
{
  short	action[MAXSGID + 1];		  // array with EPICS orders for the various signal groups; the indices are the signal-group IDs
#ifdef RBC
  short	actionNextSecond[MAXVAR];     // analog to action but for the subsequent instead of the current second in order to enable D4 to switch with a more precise time resolution than 1s
#endif
// the following parameters are not always necessary but they might be useful for tests
  short program;                      // ID of current signal program; for BALANCE and for testing reasons
  short cycleLength;                  // mainly for BALANCE
  short stage;                        // ID of desired next stage (0 if Epics wants to remain in the current stage for another second, -1 if Epics cannot optimize)
  short interstage;                   // ID of desired interstage; identical to return value 'pue' for TRENDS
  short numberMessages;               // how many messages did EPICS write during the call?
  message_t messages[MAXJSONMESSAGES];// maximum number: MAXJSONMESSAGES

// certainly also AP values will be needed; it is not yet perfectly clear which ones but they should be similar to the following:
  unsigned short APValuesGeneral[10]; // timer, tX, current stage, nr stages within TH, program, cycle length, initialized (only once), PI first step, PI second step, Balance frame valid
  unsigned short APValuesQueue[MAXSG];// the queue lengths like in Queues; the order is the order of the demands; 203 means 3 vehicles waiting at one specific demand on Sgr 2
  unsigned short APValuesTimeRequirement[MAXSG]; // the time requirements like in Queues; the order is the order of the demands
  unsigned short APValuesStageSequence[10]; // the stage sequence for the next ~100s like in StageSequence_XXX.txt, but without interstages
//  ushort APValuesSignalplan[10];    // frame handed over to VS-PLUS like in Signalplan_XXX.txt; pure VS-PLUS Array
//  ushort APValuesMessages[10];      // not needed due to array 'messages'
  char *debugLogString;               // more detailed information for debug reasons

  int callingPointNumbers[MAXDETID + 1];// connection between calling point ID and number; example: callingPointNumbers[ID = 40] = 10203

  // a more general interface is needed; see EP-104; to this end json strings are handed over; each String has the length 2*LENGTHLOGSTRING
  // The first three strings contain the Epics messages with number and parameters (only if there are some). There is one string for errors, warnings and information, respectively. 
  // The next two strings are written every second. They contain information like queue lengths, switching commands and the optimized signal plan.
  // The sixth string is written every cycle, containing e.g. saturation flows. The remaining 4 strings are empty at the moment.
  char jsonStrings[10][LENGTHDEBUGLOGSTRING];
} EPICSOutput_t;


#endif	// e_eon_h

