/* Joystick

Example code of how to read a joystick

https://www.sparkfun.com/products/9032

Craig A. Evans
7 March 2015
*/

#include "mbed.h"
#include "N5110.h"

// change this to alter tolerance of joystick direction
#define DIRECTION_TOLERANCE 0.05

// Joystick pins connections 
DigitalIn button(PTC17);
AnalogIn xPot(PTC11);
AnalogIn yPot(PTC10);

// timer to regularly read the joystick
Ticker pollJoystick;
// Serial for debug
Serial serial(USBTX,USBRX);

// create enumerated type (0,1,2,3 etc. for direction)
// could be extended for diagonals etc.
enum DirectionName {
    UP,
    DOWN,
    LEFT,
    RIGHT,
    CENTRE,
    UNKNOWN
};

// struct for Joystick
typedef struct JoyStick Joystick;
struct JoyStick {
    float x;    // current x value
    float x0;   // 'centred' x value
    float y;    // current y value
    float y0;   // 'centred' y value
    int button; // button state (assume pull-down used, so 1 = pressed, 0 = unpressed)
    DirectionName direction;  // current direction
};
// create struct variable
Joystick joystick;

int printFlag = 0;

// function prototypes
void calibrateJoystick();
void updateJoystick();

void Paddle_Move()
{
    calibrateJoystick();  // get centred values of joystick
    pollJoystick.attach(&updateJoystick,1.0/10.0);  // read joystick 10 times per second



        if (printFlag) {  // if flag set, clear flag and print joystick values to serial port
            printFlag = 0;
            serial.printf("x = %f y = %f button = %d ",joystick.x,joystick.y,joystick.button);

            // check joystick direction
            if (joystick.direction == UP)
                serial.printf(" UP\n");
            if (joystick.direction == DOWN)
                serial.printf(" DOWN\n");
            if (joystick.direction == LEFT)
                serial.printf(" LEFT\n");
            if (joystick.direction == RIGHT)
                serial.printf(" RIGHT\n");
            if (joystick.direction == CENTRE)
                serial.printf(" CENTRE\n");
            if (joystick.direction == UNKNOWN)
                serial.printf(" Unsupported direction\n");

        }


}

// read default positions of the joystick to calibrate later readings
void calibrateJoystick()
{
    button.mode(PullDown);
    // must not move during calibration
    joystick.x0 = xPot;  // initial positions in the range 0.0 to 1.0 (0.5 if centred exactly)
    joystick.y0 = yPot;
}
void updateJoystick()
{
    // read current joystick values relative to calibrated values (in range -0.5 to 0.5, 0.0 is centred)
    joystick.x = xPot - joystick.x0;
    joystick.y = yPot - joystick.y0;
    // read button state
    joystick.button = button;

    // calculate direction depending on x,y values
    // tolerance allows a little lee-way in case joystick not exactly in the stated direction
    if ( fabs(joystick.y) < DIRECTION_TOLERANCE && fabs(joystick.x) < DIRECTION_TOLERANCE) {
        joystick.direction = CENTRE;
    } else if ( joystick.y > DIRECTION_TOLERANCE && fabs(joystick.x) < DIRECTION_TOLERANCE) {
        joystick.direction = UP;
    } else if ( joystick.y < DIRECTION_TOLERANCE && fabs(joystick.x) < DIRECTION_TOLERANCE) {
        joystick.direction = DOWN;
    } else if ( joystick.x > DIRECTION_TOLERANCE && fabs(joystick.y) < DIRECTION_TOLERANCE) {
        joystick.direction = RIGHT;
    } else if ( joystick.x < DIRECTION_TOLERANCE && fabs(joystick.y) < DIRECTION_TOLERANCE) {
        joystick.direction = LEFT;
    } else {
        joystick.direction = UNKNOWN;
    }

    // set flag for printing
    printFlag = 1;
}