#ifndef BMP280_INCLUDE_H
#define BMP280_INCLUDE_H

#include "my_types.h"

/*************************** for BMP280 API ************************/
/************** I2C/SPI buffer length ******/

#define SPI_BUFFER_LEN 5
#define BUFFER_LENGTH 0xFF
#define SPI_READ  0x80
#define SPI_WRITE 0x7F
#define BMP280_DATA_INDEX 1
#define BMP280_ADDRESS_INDEX  2
#define BMP280_SPI_ADDRESS                  (0x74)

/* numeric definitions */
#define  BMP280_PRESSURE_TEMPERATURE_CALIB_DATA_LENGTH   (24)
#define BMP280_GEN_READ_WRITE_DATA_LENGTH     (1)
#define BMP280_REGISTER_READ_DELAY        (1)
#define BMP280_TEMPERATURE_DATA_LENGTH        (3)
#define BMP280_PRESSURE_DATA_LENGTH       (3)
#define BMP280_ALL_DATA_FRAME_LENGTH        (6)
#define BMP280_INIT_VALUE         (0)
#define BMP280_CHIP_ID_READ_COUNT       (5)
#define BMP280_CHIP_ID_READ_SUCCESS       (0)
#define BMP280_CHIP_ID_READ_FAIL        ((s8)-1)
#define BMP280_INVALID_DATA         (0)

/************************************************/
/**\name  ERROR CODES      */
/************************************************/
#define SUCCESS     ((u8)0)
#define E_BMP280_NULL_PTR         ((s8)-127)
#define E_BMP280_COMM_RES         ((s8)-1)
#define E_BMP280_OUT_OF_RANGE     ((s8)-2)
#define ERROR                     ((s8)-1)
/************************************************/
/**\name  CHIP ID DEFINITION       */
/***********************************************/
#define BMP280_CHIP_ID1   (0x56)
#define BMP280_CHIP_ID2   (0x57)
#define BMP280_CHIP_ID3   (0x58)
/************************************************/
/**\name  I2C ADDRESS DEFINITION       */
/***********************************************/
#define BMP280_I2C_ADDRESS1                  (0x76)
#define BMP280_I2C_ADDRESS2                  (0x77)
/************************************************/
/**\name  POWER MODE DEFINITION       */
/***********************************************/
/* Sensor Specific constants */
#define BMP280_SLEEP_MODE                    (0x00)
#define BMP280_FORCED_MODE                   (0x01)
#define BMP280_NORMAL_MODE                   (0x03)
#define BMP280_SOFT_RESET_CODE               (0xB6)
/************************************************/
/**\name  STANDBY TIME DEFINITION       */
/***********************************************/
#define BMP280_STANDBY_TIME_1_MS              (0x00)
#define BMP280_STANDBY_TIME_63_MS             (0x01)
#define BMP280_STANDBY_TIME_125_MS            (0x02)
#define BMP280_STANDBY_TIME_250_MS            (0x03)
#define BMP280_STANDBY_TIME_500_MS            (0x04)
#define BMP280_STANDBY_TIME_1000_MS           (0x05)
#define BMP280_STANDBY_TIME_2000_MS           (0x06)
#define BMP280_STANDBY_TIME_4000_MS           (0x07)
/************************************************/
/**\name  OVERSAMPLING DEFINITION       */
/***********************************************/
#define BMP280_OVERSAMP_SKIPPED          (0x00)
#define BMP280_OVERSAMP_1X               (0x01)
#define BMP280_OVERSAMP_2X               (0x02)
#define BMP280_OVERSAMP_4X               (0x03)
#define BMP280_OVERSAMP_8X               (0x04)
#define BMP280_OVERSAMP_16X              (0x05)
/************************************************/
/**\name  WORKING MODE DEFINITION       */
/***********************************************/
#define BMP280_ULTRA_LOW_POWER_MODE          (0x00)
#define BMP280_LOW_POWER_MODE              (0x01)
#define BMP280_STANDARD_RESOLUTION_MODE      (0x02)
#define BMP280_HIGH_RESOLUTION_MODE          (0x03)
#define BMP280_ULTRA_HIGH_RESOLUTION_MODE    (0x04)

#define BMP280_ULTRALOWPOWER_OVERSAMP_PRESSURE          BMP280_OVERSAMP_1X
#define BMP280_ULTRALOWPOWER_OVERSAMP_TEMPERATURE       BMP280_OVERSAMP_1X

#define BMP280_LOWPOWER_OVERSAMP_PRESSURE          BMP280_OVERSAMP_2X
#define BMP280_LOWPOWER_OVERSAMP_TEMPERATURE           BMP280_OVERSAMP_1X

#define BMP280_STANDARDRESOLUTION_OVERSAMP_PRESSURE     BMP280_OVERSAMP_4X
#define BMP280_STANDARDRESOLUTION_OVERSAMP_TEMPERATURE  BMP280_OVERSAMP_1X

#define BMP280_HIGHRESOLUTION_OVERSAMP_PRESSURE         BMP280_OVERSAMP_8X
#define BMP280_HIGHRESOLUTION_OVERSAMP_TEMPERATURE      BMP280_OVERSAMP_1X

#define BMP280_ULTRAHIGHRESOLUTION_OVERSAMP_PRESSURE       BMP280_OVERSAMP_16X
#define BMP280_ULTRAHIGHRESOLUTION_OVERSAMP_TEMPERATURE    BMP280_OVERSAMP_2X
/************************************************/
/**\name  FILTER DEFINITION       */
/***********************************************/
#define BMP280_FILTER_COEFF_OFF               (0x00)
#define BMP280_FILTER_COEFF_2                 (0x01)
#define BMP280_FILTER_COEFF_4                 (0x02)
#define BMP280_FILTER_COEFF_8                 (0x03)
#define BMP280_FILTER_COEFF_16                (0x04)
/************************************************/
/**\name  DELAY TIME DEFINITION       */
/***********************************************/
#define T_INIT_MAX          (20)
/* 20/16 = 1.25 ms */
#define T_MEASURE_PER_OSRS_MAX        (37)
/* 37/16 = 2.3125 ms*/
#define T_SETUP_PRESSURE_MAX        (10)
/* 10/16 = 0.625 ms */
/************************************************/
/**\name  CALIBRATION PARAMETERS DEFINITION       */
/***********************************************/
/*calibration parameters */
#define BMP280_TEMPERATURE_CALIB_DIG_T1_LSB_REG             (0x88)
#define BMP280_TEMPERATURE_CALIB_DIG_T1_MSB_REG             (0x89)
#define BMP280_TEMPERATURE_CALIB_DIG_T2_LSB_REG             (0x8A)
#define BMP280_TEMPERATURE_CALIB_DIG_T2_MSB_REG             (0x8B)
#define BMP280_TEMPERATURE_CALIB_DIG_T3_LSB_REG             (0x8C)
#define BMP280_TEMPERATURE_CALIB_DIG_T3_MSB_REG             (0x8D)
#define BMP280_PRESSURE_CALIB_DIG_P1_LSB_REG                (0x8E)
#define BMP280_PRESSURE_CALIB_DIG_P1_MSB_REG                (0x8F)
#define BMP280_PRESSURE_CALIB_DIG_P2_LSB_REG                (0x90)
#define BMP280_PRESSURE_CALIB_DIG_P2_MSB_REG                (0x91)
#define BMP280_PRESSURE_CALIB_DIG_P3_LSB_REG                (0x92)
#define BMP280_PRESSURE_CALIB_DIG_P3_MSB_REG                (0x93)
#define BMP280_PRESSURE_CALIB_DIG_P4_LSB_REG                (0x94)
#define BMP280_PRESSURE_CALIB_DIG_P4_MSB_REG                (0x95)
#define BMP280_PRESSURE_CALIB_DIG_P5_LSB_REG                (0x96)
#define BMP280_PRESSURE_CALIB_DIG_P5_MSB_REG                (0x97)
#define BMP280_PRESSURE_CALIB_DIG_P6_LSB_REG                (0x98)
#define BMP280_PRESSURE_CALIB_DIG_P6_MSB_REG                (0x99)
#define BMP280_PRESSURE_CALIB_DIG_P7_LSB_REG                (0x9A)
#define BMP280_PRESSURE_CALIB_DIG_P7_MSB_REG                (0x9B)
#define BMP280_PRESSURE_CALIB_DIG_P8_LSB_REG                (0x9C)
#define BMP280_PRESSURE_CALIB_DIG_P8_MSB_REG                (0x9D)
#define BMP280_PRESSURE_CALIB_DIG_P9_LSB_REG                (0x9E)
#define BMP280_PRESSURE_CALIB_DIG_P9_MSB_REG                (0x9F)
/************************************************/
/**\name  REGISTER ADDRESS DEFINITION       */
/***********************************************/
#define BMP280_CHIP_ID_REG                   (0xD0)  /*Chip ID Register */
#define BMP280_RST_REG                       (0xE0) /*Softreset Register */
#define BMP280_STAT_REG                      (0xF3)  /*Status Register */
#define BMP280_CTRL_MEAS_REG                 (0xF4)  /*Ctrl Measure Register */
#define BMP280_CONFIG_REG                    (0xF5)  /*Configuration Register */
#define BMP280_PRESSURE_MSB_REG              (0xF7)  /*Pressure MSB Register */
#define BMP280_PRESSURE_LSB_REG              (0xF8)  /*Pressure LSB Register */
#define BMP280_PRESSURE_XLSB_REG             (0xF9)  /*Pressure XLSB Register */
#define BMP280_TEMPERATURE_MSB_REG           (0xFA)  /*Temperature MSB Reg */
#define BMP280_TEMPERATURE_LSB_REG           (0xFB)  /*Temperature LSB Reg */
#define BMP280_TEMPERATURE_XLSB_REG          (0xFC)  /*Temperature XLSB Reg */
/************************************************/
/**\name  BIT LENGTH,POSITION AND MASK DEFINITION      */
/***********************************************/
/* Status Register */
#define BMP280_STATUS_REG_MEASURING__POS           (3)
#define BMP280_STATUS_REG_MEASURING__MSK           (0x08)
#define BMP280_STATUS_REG_MEASURING__LEN           (1)
#define BMP280_STATUS_REG_MEASURING__REG           (BMP280_STAT_REG)

#define BMP280_STATUS_REG_IM_UPDATE__POS            (0)
#define BMP280_STATUS_REG_IM_UPDATE__MSK            (0x01)
#define BMP280_STATUS_REG_IM_UPDATE__LEN            (1)
#define BMP280_STATUS_REG_IM_UPDATE__REG           (BMP280_STAT_REG)
/************************************************/
/**\name  BIT LENGTH,POSITION AND MASK DEFINITION
  FOR TEMPERATURE OVERSAMPLING */
/***********************************************/
/* Control Measurement Register */
#define BMP280_CTRL_MEAS_REG_OVERSAMP_TEMPERATURE__POS             (5)
#define BMP280_CTRL_MEAS_REG_OVERSAMP_TEMPERATURE__MSK             (0xE0)
#define BMP280_CTRL_MEAS_REG_OVERSAMP_TEMPERATURE__LEN             (3)
#define BMP280_CTRL_MEAS_REG_OVERSAMP_TEMPERATURE__REG             \
  (BMP280_CTRL_MEAS_REG)
/************************************************/
/**\name  BIT LENGTH,POSITION AND MASK DEFINITION
  FOR PRESSURE OVERSAMPLING */
/***********************************************/
#define BMP280_CTRL_MEAS_REG_OVERSAMP_PRESSURE__POS             (2)
#define BMP280_CTRL_MEAS_REG_OVERSAMP_PRESSURE__MSK             (0x1C)
#define BMP280_CTRL_MEAS_REG_OVERSAMP_PRESSURE__LEN             (3)
#define BMP280_CTRL_MEAS_REG_OVERSAMP_PRESSURE__REG             \
  (BMP280_CTRL_MEAS_REG)
/************************************************/
/**\name  BIT LENGTH,POSITION AND MASK DEFINITION
  FOR POWER MODE */
/***********************************************/
#define BMP280_CTRL_MEAS_REG_POWER_MODE__POS              (0)
#define BMP280_CTRL_MEAS_REG_POWER_MODE__MSK              (0x03)
#define BMP280_CTRL_MEAS_REG_POWER_MODE__LEN              (2)
#define BMP280_CTRL_MEAS_REG_POWER_MODE__REG             (BMP280_CTRL_MEAS_REG)
/************************************************/
/**\name  BIT LENGTH,POSITION AND MASK DEFINITION
  FOR STANDBY DURATION */
/***********************************************/
/* Configuration Register */
#define BMP280_CONFIG_REG_STANDBY_DURN__POS                 (5)
#define BMP280_CONFIG_REG_STANDBY_DURN__MSK                 (0xE0)
#define BMP280_CONFIG_REG_STANDBY_DURN__LEN                 (3)
#define BMP280_CONFIG_REG_STANDBY_DURN__REG                 (BMP280_CONFIG_REG)
/************************************************/
/**\name  BIT LENGTH,POSITION AND MASK DEFINITION
  FOR IIR FILTER */
/***********************************************/
#define BMP280_CONFIG_REG_FILTER__POS              (2)
#define BMP280_CONFIG_REG_FILTER__MSK              (0x1C)
#define BMP280_CONFIG_REG_FILTER__LEN              (3)
#define BMP280_CONFIG_REG_FILTER__REG              (BMP280_CONFIG_REG)
/************************************************/
/**\name  BIT LENGTH,POSITION AND MASK DEFINITION
  FOR SPI ENABLE*/
/***********************************************/
#define BMP280_CONFIG_REG_SPI3_ENABLE__POS             (0)
#define BMP280_CONFIG_REG_SPI3_ENABLE__MSK             (0x01)
#define BMP280_CONFIG_REG_SPI3_ENABLE__LEN             (1)
#define BMP280_CONFIG_REG_SPI3_ENABLE__REG             (BMP280_CONFIG_REG)
/************************************************/
/**\name  BIT LENGTH,POSITION AND MASK DEFINITION
  FOR PRESSURE AND TEMPERATURE DATA REGISTERS */
/***********************************************/
/* Data Register */
#define BMP280_PRESSURE_XLSB_REG_DATA__POS         (4)
#define BMP280_PRESSURE_XLSB_REG_DATA__MSK         (0xF0)
#define BMP280_PRESSURE_XLSB_REG_DATA__LEN         (4)
#define BMP280_PRESSURE_XLSB_REG_DATA__REG         (BMP280_PRESSURE_XLSB_REG)

#define BMP280_TEMPERATURE_XLSB_REG_DATA__POS      (4)
#define BMP280_TEMPERATURE_XLSB_REG_DATA__MSK      (0xF0)
#define BMP280_TEMPERATURE_XLSB_REG_DATA__LEN      (4)
#define BMP280_TEMPERATURE_XLSB_REG_DATA__REG      (BMP280_TEMPERATURE_XLSB_REG)
/************************************************/
/**\name  BUS READ AND WRITE FUNCTION POINTERS */
/***********************************************/
#define BMP280_WR_FUNC_PTR  s8 (*bus_write)(u8, u8, u8 *, u8)

#define BMP280_RD_FUNC_PTR  s8 (*bus_read)(u8, u8, u8 *, u8)

#define BMP280_MDELAY_DATA_TYPE u32
/****************************************************/
/**\name  DEFINITIONS FOR ARRAY SIZE OF DATA   */
/***************************************************/
#define BMP280_TEMPERATURE_DATA_SIZE    (3)
#define BMP280_PRESSURE_DATA_SIZE   (3)
#define BMP280_DATA_FRAME_SIZE      (6)
#define BMP280_CALIB_DATA_SIZE      (24)

#define BMP280_TEMPERATURE_MSB_DATA   (0)
#define BMP280_TEMPERATURE_LSB_DATA   (1)
#define BMP280_TEMPERATURE_XLSB_DATA    (2)

#define BMP280_PRESSURE_MSB_DATA    (0)
#define BMP280_PRESSURE_LSB_DATA    (1)
#define BMP280_PRESSURE_XLSB_DATA   (2)

#define BMP280_DATA_FRAME_PRESSURE_MSB_BYTE (0)
#define BMP280_DATA_FRAME_PRESSURE_LSB_BYTE (1)
#define BMP280_DATA_FRAME_PRESSURE_XLSB_BYTE  (2)
#define BMP280_DATA_FRAME_TEMPERATURE_MSB_BYTE  (3)
#define BMP280_DATA_FRAME_TEMPERATURE_LSB_BYTE  (4)
#define BMP280_DATA_FRAME_TEMPERATURE_XLSB_BYTE (5)

/****************************************************/
/**\name  ARRAY PARAMETER FOR CALIBRATION     */
/***************************************************/
#define BMP280_TEMPERATURE_CALIB_DIG_T1_LSB (0)
#define BMP280_TEMPERATURE_CALIB_DIG_T1_MSB (1)
#define BMP280_TEMPERATURE_CALIB_DIG_T2_LSB (2)
#define BMP280_TEMPERATURE_CALIB_DIG_T2_MSB (3)
#define BMP280_TEMPERATURE_CALIB_DIG_T3_LSB (4)
#define BMP280_TEMPERATURE_CALIB_DIG_T3_MSB (5)
#define BMP280_PRESSURE_CALIB_DIG_P1_LSB  (6)
#define BMP280_PRESSURE_CALIB_DIG_P1_MSB  (7)
#define BMP280_PRESSURE_CALIB_DIG_P2_LSB  (8)
#define BMP280_PRESSURE_CALIB_DIG_P2_MSB  (9)
#define BMP280_PRESSURE_CALIB_DIG_P3_LSB  (10)
#define BMP280_PRESSURE_CALIB_DIG_P3_MSB  (11)
#define BMP280_PRESSURE_CALIB_DIG_P4_LSB  (12)
#define BMP280_PRESSURE_CALIB_DIG_P4_MSB  (13)
#define BMP280_PRESSURE_CALIB_DIG_P5_LSB  (14)
#define BMP280_PRESSURE_CALIB_DIG_P5_MSB  (15)
#define BMP280_PRESSURE_CALIB_DIG_P6_LSB  (16)
#define BMP280_PRESSURE_CALIB_DIG_P6_MSB  (17)
#define BMP280_PRESSURE_CALIB_DIG_P7_LSB  (18)
#define BMP280_PRESSURE_CALIB_DIG_P7_MSB  (19)
#define BMP280_PRESSURE_CALIB_DIG_P8_LSB  (20)
#define BMP280_PRESSURE_CALIB_DIG_P8_MSB  (21)
#define BMP280_PRESSURE_CALIB_DIG_P9_LSB  (22)
#define BMP280_PRESSURE_CALIB_DIG_P9_MSB  (23)

/* right shift definitions*/
#define BMP280_SHIFT_BIT_POSITION_BY_01_BIT         (1)
#define BMP280_SHIFT_BIT_POSITION_BY_02_BITS      (2)
#define BMP280_SHIFT_BIT_POSITION_BY_03_BITS      (3)
#define BMP280_SHIFT_BIT_POSITION_BY_04_BITS      (4)
#define BMP280_SHIFT_BIT_POSITION_BY_05_BITS      (5)
#define BMP280_SHIFT_BIT_POSITION_BY_08_BITS      (8)
#define BMP280_SHIFT_BIT_POSITION_BY_11_BITS      (11)
#define BMP280_SHIFT_BIT_POSITION_BY_12_BITS      (12)
#define BMP280_SHIFT_BIT_POSITION_BY_13_BITS      (13)
#define BMP280_SHIFT_BIT_POSITION_BY_14_BITS      (14)
#define BMP280_SHIFT_BIT_POSITION_BY_15_BITS      (15)
#define BMP280_SHIFT_BIT_POSITION_BY_16_BITS      (16)
#define BMP280_SHIFT_BIT_POSITION_BY_17_BITS      (17)
#define BMP280_SHIFT_BIT_POSITION_BY_18_BITS      (18)
#define BMP280_SHIFT_BIT_POSITION_BY_19_BITS      (19)
#define BMP280_SHIFT_BIT_POSITION_BY_25_BITS      (25)
#define BMP280_SHIFT_BIT_POSITION_BY_31_BITS      (31)
#define BMP280_SHIFT_BIT_POSITION_BY_33_BITS      (33)
#define BMP280_SHIFT_BIT_POSITION_BY_35_BITS      (35)
#define BMP280_SHIFT_BIT_POSITION_BY_47_BITS      (47)
/**************************************************************/
/**\name  STRUCTURE DEFINITIONS                         */
/**************************************************************/
/*!
   @brief This structure holds all device specific calibration parameters
*/
struct bmp280_calib_param_t {
  u16 dig_T1;/**<calibration T1 data*/
  s16 dig_T2;/**<calibration T2 data*/
  s16 dig_T3;/**<calibration T3 data*/
  u16 dig_P1;/**<calibration P1 data*/
  s16 dig_P2;/**<calibration P2 data*/
  s16 dig_P3;/**<calibration P3 data*/
  s16 dig_P4;/**<calibration P4 data*/
  s16 dig_P5;/**<calibration P5 data*/
  s16 dig_P6;/**<calibration P6 data*/
  s16 dig_P7;/**<calibration P7 data*/
  s16 dig_P8;/**<calibration P8 data*/
  s16 dig_P9;/**<calibration P9 data*/

  s32 t_fine;/**<calibration t_fine data*/
};
/*!
   @brief This structure holds BMP280 initialization parameters
*/

class BMP280 {
public:
    // constructor
  BMP280 (long clock, DigitalOut * cs, SPI * bus, u8 ctrl = 0x55, u8 flt = (BMP280_FILTER_COEFF_16 << 2)){
    present = true;
    SPI_clock = clock;
    SPI_cs = cs;
    SPI_bus = bus;
    SPI_bus->format(8,0);
    SPI_bus->frequency(SPI_clock);
    ctrl_meas = ctrl;
    filter =flt;
    }
  u8 ctrl_meas;/**< chip id of the sensor*/
  u8 filter; /**< config register, mostly filter*/
  fpct zero_offs;
  unsigned long SPI_clock;
  mbed::SPI * SPI_bus;
  mbed::DigitalOut * SPI_cs;
  
  bool read_chipID(u8 expected);
  bool is_busy();
  bool present;
  void start_measure_cycle();
  void set_filter();
  void unload_calibration_parameter();
  void print_calibration_parameter(Serial *channel,const char * name);
  void print_settings(Serial *channel,const char * name);
  void unload_raw_data();
   s32 get_uncomp_temp();
  s32 get_uncomp_press();
#ifdef BMP280_ENABLE_FLOAT
  fpct compensate_temperature(s32 v_uncomp_temperature_s32);
  fpct compensate_pressure(s32 v_uncomp_pressure_s32);
  fpct temperature;
  fpct pressure;
#else
  s32 compensate_temperature(s32 v_uncomp_temperature_s32);
  u32 compensate_pressure(s32 v_uncomp_pressure_s32);
  s32 temperature;
  u32 pressure;
#endif
private:
  struct bmp280_calib_param_t calib_param;/**<calibration data*/
  u8 raw_data[6];
  u8 read_spi_8b(u8 raddr);
  void read_spi_8b(u8 reg_addr, u8 *reg_data, int cnt);
  void write_spi_8b(u8 raddr,uint8_t rdata);
  void select();
  void deselect();
};

/********************* End for BMP280 API ************************/

#endif // BMP280_INCLUDE_H



