
#ifndef VARIO_SYNTHESISER_HPP
#define VARIO_SYNTHESISER_HPP
#define VSOUND_ON   true
#define VSOUND_OFF  false

#define NSAMP 64

#include "mbed.h"

#ifndef TESTBENCH
typedef unsigned int size_t;
#endif
/**
 * This class generates vario sound.
 */
class VarioSynthesiser
{
  public:
  /**
   * The tone frequency for #min_vario.
   */
  unsigned min_frequency;

  /**
   * The tone frequency for stationary altitude.
   */
  unsigned zero_frequency;

  /**
   * The tone frequency for #max_vario.
   */
  unsigned max_frequency;

  /**
   * The number of time ticks (usually 14 Hz) for stationare altitude.
   */
  unsigned zero_timeticks;

  /**
   * The maximum number of time ticks for max_vario
   */
  unsigned max_timeticks;
  
  /**
   * The vario range of the "dead band" during which no sound is emitted
   * [cm/s].
   */
  int min_dead, max_dead;

#ifdef TESTBENCH
  VarioSynthesiser(unsigned tpl);
#else
  VarioSynthesiser(unsigned tpl, AnalogOut * s, Serial * ch);
  void set_channel (Serial * uart) {cChan = uart;}
#endif
  
  int UpDownVolume(int incr=0);
  
  bool dead_band_enabled;
  
  /**
   * tone toggles between zero_frequency and the actual lift tone
   * much like Zander varios
   */
  bool dualtone;

  
  void SetDefaults();

  /**
   * Update the vario value.  This calculates a new tone frequency and
   * other parameters
   *
   * @param vario the current vario value [m/s]
   */
  void SetVario(float vario);
   
  /**
   * Advance internal time
   *
   */
  void TimeTick();

  /**
   * Produce silence/sound from now on.
   */
  void SetSilence();
  void SetSound();

  /**
   * Enable/disable the dead band silence
   */
  void SetDeadBand(bool enabled) {
    dead_band_enabled = enabled;
  }

  /**
   * Set the base frequencies for minimum, zero and maximum lift
   */
  void SetFrequencies(unsigned min, unsigned zero, unsigned max) {
    min_frequency = min;
    zero_frequency = zero;
    max_frequency = max;
  }

  /**
   * Set the time periods for minimum and maximum lift
  void SetTimeTicks(unsigned zero, unsigned max) {
    zero_timeticks = zero;
    max_timeticks = max;
  }
*/

  /**
   * Set the vario range of the "dead band" during which no sound is emitted
   */
  void SetDeadBandRange(float min, float max) {
    min_dead = (int)(min * 100);
    max_dead = (int)(max * 100);
  }

  void SetTone(unsigned freq);
  void AckTone(unsigned freq,unsigned ticks);


  // float volume;
  int16_t ivolume;

  bool InDeadBand(int ivario) {
    return ivario >= min_dead && ivario <= max_dead;
  }
  
  bool DualtoneMode() {
    return dualtone;
  }
  
  void SetDualtoneMode(bool mode) {
    dualtone = mode;
  }
  
  float pw;

  unsigned active_freq;

  private:
#ifndef TESTBENCH
  AnalogOut * vSound;
  Serial * cChan;
  TIM_HandleTypeDef TIM6_Handle;
  DAC_HandleTypeDef hdac;
  DMA_HandleTypeDef DMA_Handle;
#endif
  
  void timerInit(int);
  void dacInit();
  void dmaInit();

  int16_t master_sound_data[NSAMP];
  uint16_t sound_data[NSAMP];
  uint16_t v_sound_data[NSAMP];


  bool mute_tone;
  unsigned time_slice;
  unsigned vario_freq;
  unsigned ack_freq;
  unsigned vario_timeticks;
  unsigned dual_tone_base_ticks;
  unsigned ticks_per_loop;
  /* used by AckTone() to remember how many time ticks remain for the ack tone
  */
  int override_vario;
  bool ack_tone_active;
  bool in_dead_band;

#ifdef TESTBENCH
  public:
#endif

  /**
   * Convert a vario value to a tone frequency.
   *
   * @param ivario the current vario value [cm/s]
   */
//  gcc_const
  unsigned VarioToFrequency(int ivario);
  unsigned VarioToTimeTicks(int ivario);

};

#endif


